﻿/*
 * ExportCurrentDocument.jsx
 * Javascript for InDesign CC, CS6
 * Version date: 20200225
 *
 * PURPOSE:
 * ========
 * Export the currently opened InDesign document to XML suitable for ePaper
 * using the "BatchXSLT for InDesign" Transformer.
 * Export the current InDesign document to 
 * -- InDesign IDML Format
 * -- all pages to PDF and JPEG 
 * -- and copy all original images
 * into the base export folder which defaults to ~/Export/in.
 * After done, a JobTicket for the transformer is written the tranformer is started.
 * The tranformer transforms/converts the files from
 * the folder ~/Export/in into the folder ~/Export/out.
 * After conversion is done, the folder ~/Export/out contains 
 * the flippable or sliding pages eBook as XML and HTML.
 * 
 * The settings from the main menu are stored in the folder sub-path 'zz_Settings/settings.set'.
 * These settings are reloaded every time this program is started.
 *
 *
 * IMPORTANT NOTES
 * ===============
 * 1. MAKE SURE, YOUR SYSTEM HAS ENOUGH PHYSICAL MEMORY!
 *    AT LEAST 8GB ! THE MORE, THE BETTER.
 * 2. MAKE SURE, INDESIGN HAS A LOT OF MEMORY AVAILABLE!
 *    Around 4GB... or it may crash.
 * 3. INSTALL ALL FONTS USED IN THE DOCUMENTS!
 *    Otherwise the InDesign documents will flow with replaced fonts.
 * 4. InDesign doument file names must end with '.indd'
 * 5. InDesign doument file names and folder names MAY NOT CONTAIN SLASHES  '/' OR BACKSLASHES  '\' AND OTHER 'UNUSUAL' CHARACTERS! Plain ASCII is safest for Internet applications.
 * 6. Crop/ Cut images to the real image content.
 *
 *
 * DISCLAIMER:
 * ===============
 * Absolutely no warranty. Use it as is or modify it to match your needs
 *
 * Author: Andreas Imhof, www.aiedv.ch
 *
 * History:
 *    20200225 - Ai: Changed to control the freeware version
 *		20200123 - Ai: Runs with InDesign 2020 on Catalina too
 *		20171025 - Ai: Fixed a problem with font conversion tables
 *		20160111 - Ai: Font conversion for ASCIII fonts to Unicode
 *		20151207 - Ai: Enhanced output file extension detection from XSLT filename
 * 		20150908 - Ai: Fixed: Clean image meta data from illegal XML characters
 * 		20150808 - Ai: pass base export folders to transformer: mainExportFolderOut/In
 * 		20150522 - Ai: new XSLT version: fixed loading font conversion tables
 * 		20141118 - Ai: set standard meta viewport: initial-scale=1
 * 		20140808 - Ai: write index.php
 * 		20140802 - Ai: Support CC 2014 v 10.x.x.x
 * 		20140708 - Ai: added extendable hooks at certain points
 *						afterInit, beforeInitMainDialog, beforeShowMainDialog, afterShowMainDialog,
 *						beforeExport, afterExport, afterExportError, 
 *						beforeDocExport, afterDocExport, 
 *						beforePageJPEGExport, afterPageJPEGExport, beforePagePDFExport, afterPagePDFExport, beforeDocPDFExport, afterDocPDFExport,
 *						beforeSplitStories, afterSplitStories, beforeTextCondVisible, afterTextCondVisible, beforeExportImages, afterExportImages, beforeIDMLExport, afterIDMLExport,
 *						beforeWriteJobTickets, afterWriteJobTickets,
 *						beforeExit
 * 		20140627 - Ai: multiple Page image sizes
 * 		20140416 - Ai: search for appl basepath on 64bit. Version updated to 2203 37
 * 		20130625 - Ai: NEW VERSION 21: support InDesign CC version 9.0
 * 		20130603 - Ai: exclude layers by name
 * 		20130404 - Ai: added pageTurnMode switch for turning pages eBook
 * 		20130208 - Ai: worked around a problem, where images are named like page jpegs: adding to page JPEGs and PDFs the prefix '_px'
 * 		20120909 - Ai: added extract of image meta data
 * 
 */

// ****** Configurable SETTINGS ******
var copyright = "\u00a9www.AiEDV.ch";
var version = "40.00 46";   // Transformer and Scripts version
var packageExtension = "";        // set to an extension like 'V6', special editions like '_PE' or 'Vxx' to a dedicated package installation or empty for the standard package installation
                                    // this expands the main application installation path to
                                    // BXSLT4InDesignBasePath = "~/BatchXSLT4InDesign" + packageExtension + "/BatchXSLT/"
                                    // as well as the communications folder to
                                    // jobticketFilePath = "~/BatchXSLT4InDesignComm" + packageExtension + "/Scripts/zz_Settings"
var tranformerName = "BatchXSLT";
var applicationHumanName = tranformerName + " for InDesign";
var applicationProgramName = tranformerName + "4InDesign";
var folderSeparator = "/";	// default to OSX
var os = "";
var programFolder = "";
var BXSLT4InDesignAppl = tranformerName + ".app";
getOS();

var initFileName = tranformerName + "InitExportCurrentDocument" + packageExtension + ".ini";  // file to initialize settings path
var initFileNameLoaded = "";        // the file path name loaded to initialize this export script
var mainExportFolder = "~/Export";  // The default path for XML data exported from InDesign
                                    // (a path starting with tilde '~' points to the users home directory)
                                    // -- set it to any existing path which should act as default export folder
                                    //    and no folder chooser dialog will be shown
                                    // -- set to "" to show a dialog to choose the export folder

var mainExportFolderInTransformer = "";		// if the transformer runs on a different machine, the machine specific data input path can be set here
var mainExportFolderOutTransformer = "";	// samo for out path



var dontAutoSaveSettingsFile = false;  // if set to true, the settings changes will never be written back to disk
var settingsFilePathDefault = "~/" + applicationProgramName + "Comm" + packageExtension + "/Scripts/zz_Settings";	// define default settings path
var settingsFilePath = settingsFilePathDefault;	// we must store settings somewhere into user home to have write access
var settingsFileNameDefault = "settings.set";  // 	// define default settings file name. default = "settings.set"
var settingsFileName = settingsFileNameDefault;  // name of the settings file to load. default = "settings.set"
var settingsFilePathAlt = "zz_Settings";							// if we have a problem try InDesign Application's script folder.
var utilitiesFilePath = "zz_Utilities";
var initLoaded = loadInitFromFile();
//alert("After loadInitFromFile()\ninitLoaded: " + initLoaded + "\nsettingsFilePath: " + settingsFilePath + "\nsettingsFileName: " + settingsFileName);

                                        // The path to the folder where InDesign and the transformer will communicate 
var BXSLT4InDesignCommDirDefault = "~/" + applicationProgramName + "Comm" + packageExtension + "/" + tranformerName + "/";
var BXSLT4InDesignCommDir = BXSLT4InDesignCommDirDefault;

var BXSLT4InDesignBasePathDefault = "~/" + applicationProgramName + packageExtension + folderSeparator + tranformerName + folderSeparator;	// use this as default because user might want to test a new version and can copy it into the user's home directory
																		// instead of Apllication folder which actually should be the default
var BXSLT4InDesignBasePathDefault2 = programFolder + folderSeparator + applicationProgramName + packageExtension + folderSeparator + tranformerName + folderSeparator;	// without the root/ProgramFilesFolder
var BXSLT4InDesignBasePathFromPrefs = BXSLT4InDesignBasePathDefault;	// might be overwritten by loadCommPrefsFromFile()
loadCommPrefsFromFile();
var BXSLT4InDesignBasePath = BXSLT4InDesignBasePathFromPrefs;

var BXSLT4InDesignPath = BXSLT4InDesignBasePath + BXSLT4InDesignAppl;

var BXSLT4InDesignApplRunsOnWindows = "";        // default = "" : tranformer runs on the same machine as this export script. "1" means, that tranformer runs on a Windows remote machine
var BXSLT4InDesignApplRunsOnOSX = "";            // default = "" : tranformer runss on the same machine as this export script. "1" means, that tranformer runs on a OSX remote machine
var BXSLT4InDesignApplRunsOnUnix = "";           // default = "" : tranformer run on the same machine as this export script. "1" means, that tranformer runs on a Unix remote machine
var servernameOnWindows = "\\\\servername\\";    // If one of the above "BXSLT4InDesignApplRunsOnXXXXX" is set to 1 - Export script and tranformer are running on different machine types 
var servernameOnOSX = "/Volumes/";               //       the "servernameOnXXXX" variables must be set to know how path names must be converted, so they can be understood by the target system running the tranformer
var servernameOnUnix = "//servername/";          //       Example: this export script is running on OSX and tranformer is running on Windows the these vaiables are used:
                                                 //             BXSLT4InDesignApplRunsOnWindows = "1";
                                                 //                 and the paths are converted using 
                                                 //             servernameOnOSX = "/Volumes/";
                                                 //             servernameOnWindows = "\\\\dataserver\\";
                                                 //                 to convert a OSX path: /Volumes/dataserver/DATA
                                                 //                 to a Windows path:     \\dataserver\DATA

var BatchXSLTtriggerMode = 0;	// how to trigger tranformer
								// 0 = normal: Export and then trigger
								// 1 = don't trigger and don't write JobTickets
								// 2 = trigger only, no doc export
//var BXSLTTriggerFile = "_BXSLT_trigger.go";	// starting tranformer's transform process with a trigger file
var BXSLTTriggerFile = "";						// starting tranformer's transform process with NO trigger file
												// set to "" to not to start transformer with a trigger file
var doStartBatchXSLT = true;					// default = true to start transformer (on the local machine)
												// set to false to not start transformer application (just communicate through BXSLT4InDesignCommDir. tranformer is running on an other machine)
var doControlBatchXSLT = true;					// default = true to control transformer by JobTickets (on the local or remote machine) at the BXSLT4InDesignCommDir path
												// set to false to not to write JobTickets for the transformer application. tranformer is running on a machine which is not reachable

var companyName = "";     // adds the company name to the export path
var objectName = "";      // adds the object name to the export path
var objectShortcut = "";  // adds the object shortcut to the export path
var issueYear = "";       // adds the issue year name to the export path
var issueDate = "";       // adds the issue date to the export path
var fileDateFormat = "";  // the format string if NOT YYYYMMDD to retrieve a document's date like:
                          // DDMMYY or YYDDMM
var issueDateFormatStringList = ["YYYYMMDD", "YYYYDDMM", "YYMMDD", "YYDDMM", "DDMMYY", "DDMMYYYY", "YYYY-MM-DD", "YYYY-DD-MM"];	// YYYYMMDD, YYYYDDMM, YYMMDD, YYDDMM, DDMMYY, DDMMYYYY
var cssName = "";         // if empty, the css name is generated from the document name
                          // if name is set like 'NYT.css', a css with this name is created if not already exists in the ../........../XSLCSS/ folder
var CSSclassNamesASCII = 0;		// 0 = default = allow UTF-8 in CSS class names, 1 = make plain ASCII class names
var CSSfontFamilyString = "";	// empty to write actual font into css, or a font-family css string like "Verdana","Arial",sans-serif (without ending semicalon or space)
var xslcss_relpath_override = "";	// fixed relative path for XSLCSS folder like "../../../XSLCSS/" (with ending slash) or empty to calculate
var showExportFolder = true;            // true to show export folder in finder

var splittingStoriesOption = 0;         // 0 = don't split chained story frames
                                        // 1 = split chained story frames when changing page
                                        // 2 = split ALL chained story frames

var closeDocumentAfterDone = false;     // close document after done
var allowRevertDocument = true;         // allow to revert to last saved version after story splitting (CS3 and later only)

var showAllConditionalText = false;      // set to true, to make all conditional text visible before exporting to INX (not in PDF and page JPEGs) (CS4 and later only)


var showMessage = true;                 // true to show completion message(s), false for no message
var silentMode = 0;                     // Flags to combine for desired Messages
                                        // -1 = absolutely no message and dialog
                                        // 0 = show main export dialog and processing and completion messages
                                        // 1 = no main dialog
                                        // 2 = no error message (silently quit)
                                        // 4 = no processing messages
                                        // 8 = no completion message
                                        // 128 = silentMode already preset - do not alter
var saveSettings = true;                // true to save settings when starting export
var overwriteExistingFiles = false;     // true to always recreate INX, pages PDF and JPEG
                                        // false to check if they exist. if YES continue with next
var noHTMLconversion = "0";             // 1 to NOT to create a HTML equivalent to XML file

                                        // The name of the InDesign IDML/Interchange Format export
                                        // exactly as it appears in InDesign's Export popup menu.
var inxExportFormatStringCS5 = "INDESIGN_MARKUP";	// CS5 -like
var inxExportFormatStringCS3 = "INTERCH";			// CS3 -like
var inxExportFormatStringCS2 = "INTERCH";			// CS2 -like
var inxExportFormatString = "";         // preset to automatically detect

                                        // The full path to the transformer application: 
                                        // (a path starting with tilde '~' points to the users home directory which is default)
                                        // -- for OSX systems: anypath/tranformer.app
                                        // -- for Windows: anypath\tranformer.bat
                                        // -- set to "" to not to start Tranformer
                                        // NOTE:
                                        // this usually is overridden by loadCommPrefsFromFile()

var runBeforeTransformApp = "";         // command tranformer to run an external Application or Script before a transform is started
                                        // called by an application caller string, a comma separated string 'applString' described below:
                                        // applString ="app_path,params,envir,workdir,convertParamsToNativeCharset,waitComplete"
                                        // where:
                                        // app_path = full path/name to application to call
                                        // params = space separated list of parameters
                                        // envir = a "+++" separated list of environment variables to set like path=mypath+++var1=myvarstring1+++var2=myvarstring2
                                        // workdir = path of working directory to set
                                        // convertParamsToNativeCharset = convert paths to native charset or not
                                        // waitComplete = "0" to NOT to wait for application completed, otherwise: wait for completion
                                        // 
                                        // multiple application commands can be defined separated by '/#/'
var runAfterTransformApp = "";          // command tranformer to run an external Application or Script after a transform has completed
var runBeforeJobApp = "";               // command tranformer to run an external Application or Script before any JobTicket stuff is done
var runAfterJobApp = "";                // command tranformer to run an external Application or Script all JobTicket stuff is done
                                        // description see above 'runBeforeTransformApp'


	// how to set up magnifier glass (flip page book only):
	// Option 0, value: 0 = do not enable, allow normal mouse over only
	// Option 1, value:10 or 1 = enable glass and set default view mode with floater and allow toggle
	// Option 2, value:11 or 111 = enable glass and set default view mode with magnifying glass and allow toggle
	// Option 3, value:110 = enable glass and set view mode with magnifying glass and DO NOT allow toggle
var magnifyingGlass=0;

	// For custom output transform, state the xsl to use like for ex. "xml2htmlList_CC.xsl" to create a dedicated HTML article list
	// this will override the output mode pop-up
var outputModeCustomXSL = "";

var enableXMLtagsMenu = false;	// EXPERIMENTAL XML Tags export: show/hide the checkbox in main export menu
var includeXMLtags = 0;			//                               1 to include custom XML tagging info, otherwise 0

var fontsizeFactor = 1.0;	// resize fonts in CSS. default is original size 1.0
							// 0.75 seems to be a usable value to have the font size in a browser shown in same size like the printed document
var fontsizeMinimum = 0.0;	// allow minimum font size like 8.0 or set to 0.0 to leave original size
var fontsizeUnits = 0;		// default = 0. /* in test phase: set to 1 to export fontsizes in percent to fontsizeBase, 2 = em */
var fontsizeBase = 9.0;		// default = 9.0 (for 9.0pt). Used as 100% if fontsizeUnits is set to 1 or 2 /* in test phase too */

var groupBoxesToArticles = 1;	// 1 = default: FOR BROWSER VIEW ALWAYS 1!! - 1 = make article groups, 0 = don't. set to 1 when article groups are wanted
var sortBoxesByAreaSize = 1;	// 1 = default: 1 = smaller boxes in front, 0 = don't sort. set to 1 when article should be sorted by area size
var sortBoxesByYposition = 1;	// 1 = default: 1 = box top/down sorting, 0 = don't. set to 1 when boxes should be sorted top down
var sortBoxYtolerance = 1.5;	// set to 0.0 to use strict top-down sorting
								// set to x.x to check for title box height too
var imageDoROTATE = 0;			// 0 = default: 1 = rotate the converted images, 0 = do not rotate
var imageEXCLUDE = "excl_;555";			// exlude images from export when name starts with a certain string. default = "excl_;555" a semicolon separated list of chars
var excludeHiddenLayers = 1;	// 1 = default = suppress invisible layers, 1 = export all layers
var excludeLayers = "noexp";	// "noexp" = default = suppress elements on layer named "noexp". comma separated list of layers to exclude
var suppressAllEmptyBox = 0; 	// 0 = default = export all empty boxes, 1 = suppress them
var suppressEmptyTextBox = 0; 	// 0 = default = export empty text boxes, 1 = suppress them
var suppressEmptyImageBox = 0; 	// 0 = default = export empty image boxes, 1 = suppress them
var suppressUnassignedBox = 0; 	// 0 = default = export unassigned content boxes, 1 = suppress them
var suppressLineBox = 1; 		// 0 = default = export line boxes, 1 = suppress them
var suppressNonPrintingBox = 1;	// 1 = default = suppress non printing boxes, 0 = export them
var pageframingBoxNoExport = 12;	//default = 0, otherwise: don't export large boxes combinable flag: 1 = text, 2 = images, 12 = both
var pageframingBoxSizeThreshold = 1.0; 	// default = 1.0, the size factor of page size when a box should be treated as large page framer
var pageframingBoxesNoCatch = 1;	//  >0 if large page framing boxes should not catch other boxes to article groups

var delete_individual_bookpagesXML = 1;	// default = 1, delete all indivial pages XML files of a book after having merged them, 0 = don't delete
var delete_individual_bookpagesHTM = 1;	// default = 1, delete all indivial pages HTM files of a book after merge, 0 = don't delete

var metaViewportInitialScale = 1;	//  = Math.floor(400/480)
var metaViewportInitialScalePageWidthBase = 400;
var metaViewportInitialRescaleFactor = metaViewportInitialScalePageWidthBase * metaViewportInitialScale;
													// will be calculated as: metaViewportInitialScalePageWidthBase * metaViewportInitialScale / pageImageWidth;
													// the initial-scale based on 400px. other page (pageImageWidth) in pixels will be resized based on this factor like:
													//  330 * 0.68 / 420 for a page width of 420 pixels
													// set to 0 to not change initial-scale factor for other page sizes
var metaViewportWidth = "device-width";
var metaViewportUserScalable = "yes";
var metaViewportContent ="initial-scale=" + metaViewportInitialScale + ", width=" + metaViewportWidth + ", user-scalable=" + metaViewportUserScalable;	// a viewport meta tag's 'content' attribute value
								/* The content attribute of a 'viewport' meta tag like: "initial-scale=0.68, width=device-width, user-scalable=yes"
									&lt;meta content="initial-scale=0.68, width=device-width, user-scalable=yes" name="viewport"&gt;
									will add the meta tag: name="viewport" content="initial-scale=0.68, width=device-width, user-scalable=yes"
								*/

					// a list of keyword=content*#*keyword=content*#*....
var websiteParams = "logoName=*#*logoURL=*#*logoTitle=*#*logoURLtarget=";
					// 'logoName'= the name of the logo file (relative to CSS or absulute URL)
					// 'logoURL' = the URL to open when clicked on logo
					// 'logoTitle'= the titlw to show on mouseover
					// 'logoURLtarget'= the window name to open the clicked logoURL: _blank, _top, _parent, _self

var addVirtualKeyboard = 0;	// set to 1 to ad a virtual keybord to input fields for systems without a real keyboard otherwise 0

var totalencode = false;		// set to true to encode all characters above ASCII with encoded hex values as xXX
										// especially useful if filenames contain CJKV (Chinese, Japanese, Korean,Vietnames) characters
										// and the web server is not able to handle such names.
										// Also when exporting on Windows, this flag might be necessary to be set to true -->

var externalProcessTimeout = 180000;	// timeout for external processes. For ImageMagick set to 3 minutes
var continueXMLparseOnFatalError = 0;	// XML parser should try to continue after a fatal XML document error like non-wellformedness ( use with CAUTION )

// ****** END OF Configurable SETTINGS ******

var exportSubPath = "";	// the sub-path below all 'mainExportFolderXXX' folders
						// built from issue Company Name/object shortcut/Year/Date
var suppress_exportSubPath = false;	// suppress any given exportSubPath
var jtSubFolder = "";	// the sub-path to data for a certain jobticket: INDDin/folder
var currentDocumentFilePathName = "";	// the path and name to the actually opened document file in batch mode - it may be an unsaved document

var mainDialogErrorMessage = "";
var mainDialogErrorCode = 0;
var exportINX = 1;	// 0 = don't export XML for DEBUG only
var outputMode = 0;	// set default to flip page eBook
var pageTurnMode = 'turn';	// set default to turning pages eBook
							// 'turn' = turning pages
							// 'slide' = sliding pages
var exportImages = 1;	// 0 = don't export images, 1 = JPEG, 2 = PNG [ only when ImageMagicvk installed, 3 = GIF, 4 = TIFF]
var exportPageJPEGs = true;
var exportPagePDFs = true;
var exportPagePDFusePreset = null;			// a pdfExportPreset object to use for single page PDFs
var exportPagePDFusePresetName = "";		// the name of pdfExportPreset object
var exportPagePDFusePresetIndex = -1;		// the index of pdfExportPreset object - default to invalid
var exportDocumentPDF = false;
var exportDocumentPDFusePreset = null;		// a pdfExportPreset object to use for document PDFs
var exportDocumentPDFusePresetName = "";	// the name of pdfExportPreset object
var exportDocumentPDFusePresetIndex = -1;	// the index of pdfExportPreset object - default to invalid
var forcePDFinteractivity = true;			// force activation of interactive elements
var thePDFdocname = "";
var createFakeDocumentPDF = false;			// set to true to create a 'fake' document PDF file with zero bytes instead of actually calling InDesign to create it
											// InDesign has problems (memory leaks, crashes) when creating huge PDF files of the entire document
											// The fake zero bytes file named like "MYdoument_doc.pdf" can later be replaced by the real document PDF
var exportDocumentPDFspreads = false;
var exportPROFeatures = false;

// get PDf export presets
var exportPDFpresets = null;		// all available pdfExportPresets
var exportPDFpresetsNames = null;	// all names available pdfExportPresets
try { exportPDFpresets = app.pdfExportPresets; } catch(e){}
if (exportPDFpresets != null) {
	exportPDFpresetsNames = new Array();
	for (var i = 0; i < exportPDFpresets.length; i++) {
		exportPDFpresetsNames[exportPDFpresetsNames.length] = exportPDFpresets[i].name;
	}
	// set defaults
	if (exportPDFpresetsNames.length > 0) {
		exportPagePDFusePresetIndex = 0;		// default to first preset (usually [High Quality Print])
		exportPagePDFusePresetName = exportPDFpresetsNames[exportPagePDFusePresetIndex];
		exportDocumentPDFusePresetIndex = 0;
		exportDocumentPDFusePresetName = exportPDFpresetsNames[exportDocumentPDFusePresetIndex];
		//alert(exportPDFpresets + "\nnumitems:" + exportPDFpresets.length + "\nnames:" + exportPDFpresetsNames + "\n\nset name:" + exportDocumentPDFusePresetName + "\n\nset index:" + exportDocumentPDFusePresetIndex);
	}
}

var theBaseDocumentName = "";	// document name with replaced illegal characters
var TABLE_CELLS_WIDTH_PRESERVE = "1.3";	// factor to resize HTML table cell widths
var TABLE_BORDER_COLLAPSE = "1";	// merge table gap to one line
var TABLE_AS_BLOCK = "0";			// set to 1 to show tables always as block elements (on new line)

var imageAddTargetPrefix = 1;			// 1 = add the unique box id as target image name prefix, 0 for no prefix
var imageOverwriteTargetIfExists = 1;	// 1 = always recreate the target image even if it already exists, 0 to not to overwrite existing target image
var imageMissingCreateFromPagePreview = 2;	// (without ImageMagick: default = 2, WITH ImageMagick: default = 1)
                      // set to 0 to never recreate images from page preview
											// set to 1 (default) to extract images from screen page when original image file is missing (image box ectracted by InDesign script)
											// set to 2 to always create from screen page (image box ectracted by InDesign script)
											// set to 3 to extract images from page preview JPEG if original image file is missing (extrtacted/cut by transformer)
											// set to 4 to always create from page preview (extrtacted/cut by transformer)
var extractImageMetaData = 0;	// set to 1 to extract an image's meta data if available
var fullImageMetaData = 0;		// set to 1 to extract full meta data data not only title and description
var typeImageMetaData = "--metadata";	// type of meta data:
										// --metadata: ALL metadata as text lines: Description: Description text äöüCR
										// --xmp: LESS info as xmp XML
										// --json; ALL meta as json array: { "description":"Description: Description text äöü.", "xyz":"xyztext", ..., }

var image_COLORPROFILE = "";	// empty is -colorspace sRGB. or 2 -profile parameters fro input and output
var imageInputParam = "";		// image input parameters when IM is reading in the original image like: -density 300
								// High quality transparent PNG: -density 300 -background none -transparent black
								// Midle quality transparent PNG: -density 150 -background none -transparent black
var imageJPEGembedColorProfile = false;	// true = include color profile in image jpeg, otherwise false
var imageInputDPI = "150";		// image input density to read on canvas
var imageDPI = "150";			// image output density
								// Normal quality: 96
								// High quality: 300
								// Middle quality: 150
var imageQUALITY = "90";		// image output JPEG quality
var imageParameters = "1.0";	// image export options for max 2 images (in differen sizes) separated by //
								// syntax: optionsImage1//optionsImage2
								// where optionsImageX:
								// scaleValue[;ImageMagick parameters[;nocrop]]
								// where:
								// scaleValue: a) a factor like 1.0 for original size or 2.0 to double size
								//             b) a fixed width like 125x for 125 pixels in width
								//             c) a fixed height like x200 for 200 pixels
								// ImageMagick parameters: like "-sharpen 1" or "-modulate 105 -strip"
								// nocrop: to use original image and not the one cropped by InDesign box
								// example:
								// scale by 1.5; enhance contrast and brightness and strip profiles; from original image
								//        1.5;-contrast -modulate 110 -strip;nocrop
								// fixed width 150px; sharpen and enhance brightness and strip profiles from  cropped image//second image 3 times larger; enhanced contrast
								//        150x;-sharpen 1 -modulate 110 -strip//3.0;-contrast
var imageOutputFinishParam = "-strip";	// output image finishing parameters like "-strip" to strip all comments and profiles from output image
var copyOriginalImageFiles = 0;	// set to 1 to copy ALL original files to output folder, 2 to copy pdf, jpeg, gif and png
var imageKeepTransparency = 0;	// set to 0 to have white background. 
var imageTransparentColor = "black";	// which color should be converted to transparent like 'black' #00000000
var imgCnt = 0;					// total number of images in this document
var imgMissing = 0;				// how many source images are missing
var imgOK = 0;					// how many source images are available

var pageImageDPIOriginal = 300;		// minimal DPI of the original page preview image initially exported from InDesign at original page size
var pageImageQualityOriginal = JPEGOptionsQuality.HIGH;	// the JPEG quality. possible values:
														// JPEGOptionsQuality.LOW
														// JPEGOptionsQuality.MEDIUM
														// JPEGOptionsQuality.HIGH
														// JPEGOptionsQuality.MAXIMUM

var pageImageWidth = "500//800";		// two resized page preview images for the browser
var pageImageDPI = "300";
var pageImageQuality = "90";
var pageJPEGParams = "";	// input page image parameters like "-page...."
var pageJPEGfinish = "";	// page image finishing parameters like "-brightness-contrast 50x23" to adjust brightness and contrast of output page image
var pageJPEGembedColorProfile = false;	// true = include color profile in page jpeg, otherwise false

var catchRadius = 0;
var allBoxesNoCatch = 0;         // combinable flags for box type catching: 0=normal catching, 1=all don't catch, 2=emptyBoxesNoCatch, 4=imageBoxesNoCatch, 8=chainedBoxesNoCatch, 16=textBoxesNoCatch, 32=chainedFirstBoxesNoCatch, 64=chainedFollowingBoxesNoCatch
var unassignedBoxesNoCatch = true;
var emptyBoxesNoCatch = false;
var textBoxesNoCatch = false;
var chainedBoxesNoCatch = true;
var chainedFirstBoxesNoCatch = true;
var chainedFollowingBoxesNoCatch = true;
var imageBoxesNoCatch = false;
var suppressExportMouseOvers = "8";	//	addable flags to suppress export and mouse overs in output html flipbook web site (processed by flipbook.xsl)
									//	1 to completely suppress the mouse over function
									//	2 to suppress all text and the mouse over text
									//	4 to suppress all images and the mouse over images
									//	8 to suppress mouse over ALL empty boxes
									//	add each flag as it own char like: '14'


var	appVersion = app.version;
var majorINDVersion = 7;	// assume CS5
var is_CS3_or_later = true;
var is_CS4_or_later = true;
var BXSLTstarted = false;

var BatchXSLTMajorVersion = "";
var BatchXSLTMinorVersion = "";
var BatchXSLTRequiredXslVersionName = "";

var expectedXMLfilePath = "";	// the resulting XML file and where we should find it

var hDs="0123456789abcdef";

// ERROR codes
var last_errorCode = 0,
	error = {
		text : "",
		lastcode : 0,
		NOERR : 0,
		WRONG_INDD_VERSION : -1,
		EXPORTPATH_NOT_FOUND : -2,
		NO_OPEN_DOC : -3,
		INXFORMAT_NOT_DETECTABLE : -4,	// IDML format name not detectable
		UNSAVED_DOC : -5,
		EXPORT_FOLDERSTRUCT_NOT_FOUND : -6,
		EXPORT_FOLDERSTRUCT_NOT_CREATED : -7,
		NO_EXPORT_FOLDER : -8,
		COULD_NOT_START_BATCHXSLT : -9,
		MASTER_OUTPUTXSL_NOT_FOUND : -10,
		OPENENIG_DOC_FAILURE : -11,
		EXPORT_CANCELLED : -12,
		FOLDER_CREATE : -50,
		FILE_CREATE : -51
	};


var scriptsPath = getScriptsPath(),
	scriptName = getScriptName();

var useHooks = true,			// do check for hooks to call
	hooksBasePath = scriptsPath + folderSeparator + "zz_Plugins" + folderSeparator + "hooks" + folderSeparator,			// the base path to all hooks folders
	hooksPath = "",				// the path to certain hooks folders
	hooksFolder = null,			// the hooks folder object
	hooksScripts = null,		// the hooks folder scripts files list
	hooks = {
		/* object of this type:
		"settingsFileName": {		// hook for certain settings
								"callerscriptname":scriptName,	// the caller script name
								"hook":"hook name",				// the hook name
								"args":null,	// arguments passed to scripts called by doScript (the doScripts arguments param is not working)
												// can be of any type, the callee must know how to handle it. usually an object.
								"retval":0		// the return value set by a script called by doScript (the doScripts does not return anything)
							}
		*/
	},

	docExportRun = 0,
	exportReRun = true;

// STATUS codes
var last_statusCode = 0;
var status = {
		OK : 0,
		EXPORTED_DOC : 1,
		NOTHING_TODO : 2
	};
var callerScript = "";

// Are we called from a caller-script and have to get parameters?
var haveCallerScript = false;
var isBatchMode = false;
try {
	if ( app.scriptArgs.isDefined("settingsFilePath") ) {
		settingsFilePath = app.scriptArgs.getValue("settingsFilePath");
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("settingsFileName") ) {
		settingsFileName = app.scriptArgs.getValue("settingsFileName");
		haveCallerScript = true;
	}
} catch(e) {}

// determine a writable settings folder (in application or in home)
if (!haveCallerScript) settingsFilePath = settingsFilePathWritable();
// copy all settings files from InDesign's scripting folder into the writable communications folder if not already there
var settingscopied = copySettingsFiles();

// load previously saved settings
var settingsloaded = loadSettingsFromFile();

// override more settings from caller script
try {
	if ( app.scriptArgs.isDefined("callerScript") ) {
		callerScript = app.scriptArgs.getValue("callerScript");
		haveCallerScript = true;
		if (callerScript.toLowerCase().indexOf("batch") >= 0) isBatchMode = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("silentMode") ) {
		silentMode = parseInt(app.scriptArgs.getValue("silentMode"));
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("exportSubPath") ) {
		exportSubPath = app.scriptArgs.getValue("exportSubPath");
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("suppress_exportSubPath") ) {
		if (app.scriptArgs.getValue("suppress_exportSubPath") == "true") suppress_exportSubPath = true;
		else suppress_exportSubPath = false;
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("mainExportFolder") ) {
		mainExportFolder = app.scriptArgs.getValue("mainExportFolder");
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("jtSubFolder") ) {
		jtSubFolder = app.scriptArgs.getValue("jtSubFolder");
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("xslcss_relpath_override") ) {
		xslcss_relpath_override = app.scriptArgs.getValue("xslcss_relpath_override");
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("BatchXSLTtriggerMode") ) {
		BatchXSLTtriggerMode = parseInt(app.scriptArgs.getValue("BatchXSLTtriggerMode"));
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("noHTMLconversion") ) {
		noHTMLconversion = app.scriptArgs.getValue("noHTMLconversion");
		haveCallerScript = true;
	}
} catch(e) {}
try {
	if ( app.scriptArgs.isDefined("currentDocumentFilePathName") ) {
		currentDocumentFilePathName = app.scriptArgs.getValue("currentDocumentFilePathName");
		haveCallerScript = true;
	}
} catch(e) {}

try {	// clear script parameters now as the process could be manually aborted and args would still be set.
	app.scriptArgs.clear();
} catch (e) {}

var mainExportFolderInConst = mainExportFolder + folderSeparator + "in";
var mainExportFolderIn = mainExportFolderInConst;
var mainExportFolderOutConst = mainExportFolder + folderSeparator + "out";
var mainExportFolderOut = mainExportFolderOutConst;
var mainExportFolderDoneConst = mainExportFolder + folderSeparator + "done";
var mainExportFolderDone = mainExportFolderDoneConst;

build_mainExportFolders();

/*
if (haveCallerScript && true) {	// debug only
	alert("Variables from app.scriptArgs:"
	+ "\ncallerScript: " + callerScript
	+ "\nsilentMode: " + silentMode
	+ "\nexportSubPath: " + exportSubPath
	+ "\nsuppress_exportSubPath: " + suppress_exportSubPath
	+ "\nmainExportFolder: " + mainExportFolder
	+ "\njtSubFolder: " + jtSubFolder
	+ "\nxslcss_relpath_override: " + xslcss_relpath_override
	+ "\nnoHTMLconversion: " + noHTMLconversion
	+ "\ncurrentDocumentFilePathName: " + currentDocumentFilePathName
	+ "\n--------- new paths:"
	+ "\nmainExportFolderIn: " + mainExportFolderIn
	+ "\nmainExportFolderOut: " + mainExportFolderOut
	+ "\nmainExportFolderDone: " + mainExportFolderDone
	);
}
*/


var lg = new Array();		// the array holding all menu and message strings
var appLanguageCode = "en";	// english
appLanguageCode = getAppLanguageCode(app.locale);


//appLanguageCode = "en";	// uncomment to override the InDesign application language
var languageFileName = getScriptsPath() + folderSeparator + utilitiesFilePath + folderSeparator + "localized" + folderSeparator + "ECDlocal_" + appLanguageCode + ".jsx";
var languageFile = new File(languageFileName);
if (languageFile.exists == false) {
	languageFileName = getScriptsPath() + folderSeparator + utilitiesFilePath + folderSeparator + "localized" + folderSeparator + "ECDlocal_en.jsx";
	languageFile = new File(languageFileName);
	languageFile.encoding = "UTF-8";
	if (languageFile.exists == false) {	// FATAL: no menu language file found
		alert("FATAL ERROR! No English language file found in the InDesign's Scripting path:\n" + languageFile.fsName + "\n\nReinstall the exporter scripts!\nProcessing aborted!");
		exit();
	}
}
// load the localized file
app.doScript(languageFile,ScriptLanguage.javascript,[lg,applicationHumanName,tranformerName,BXSLT4InDesignCommDir]);


var numSplittedStoryBoxes = 0;

var windowTitle = applicationHumanName + " - XML Export  v" + version + "    '" +  settingsFileName + "'    " + copyright;

var availableOutputXSLs = new Array();


//**************************************
// initialize and start export
var orig_userInteractionLevel = app.scriptPreferences.userInteractionLevel;
app.scriptPreferences.userInteractionLevel = UserInteractionLevels.interactWithAll;

var numDocsOpen = app.documents.length,
	numBooksOpen = app.books.length,
	doingBookExport = false,
	currentBook = null, currentBookContent = null, currentBookName = "", currentBookPath = "",	// name  and path of current book
	exportBookDocumentPDF = false,	// set to true when the book document PDF must be created
	success = -1;




// allow hook to modify initialization
runHooks(scriptName,"afterInit", null);

while (exportReRun == true) {
	exportReRun = false;
	docExportRun++;
	// run hooks before exporting
	runHooks(scriptName,"beforeExport", null);
	if (getHookretval(settingsFileName) != -1) {	// skip export if -1 is returned
		if ((numDocsOpen > 0) || (numBooksOpen > 0) || ((BatchXSLTtriggerMode & 2) == 2)) {
			if ((numDocsOpen > 0) || ((BatchXSLTtriggerMode & 2) == 2)) {	// export a single document (if one is open)
				success = initBXSLTExport();
			}
			else {	// export documents of a book
				success = doBookExport();
			}
		}
		else {	// nothing ready to export
			alert(loc(5));	// open document or book ...
		}
	}

	//alert("success: " + success);
	// run hooks only if export was not cancelled by user
	if (success == 0) {
		runHooks(scriptName,"afterExport", null);
	}
	else {
		runHooks(scriptName,"afterExportError", {"success":success});
	}
}


try {	// customer reported Javascript error number 264 "Can not open Resource File" after all work was done.
		// seemed to be temporary InDesign problem with resources
	app.scriptPreferences.userInteractionLevel = orig_userInteractionLevel;
} catch (e) {}
//alert("settingsFileName: " + settingsFileName + "\nsilentMode: " + silentMode);
try {	// due to above reported error we also catch this...
	app.scriptArgs.clear();
} catch (e) {}
try {	// due to above reported error we also catch this...
	app.scriptArgs.setValue("ECD_exitcode", ""+success);
} catch (e) {}


// run hooks before exit
runHooks(scriptName,"beforeExit", {"success":success});


exit();

/* ==================== FUNCTIONS ==================== */

/****************************
 * get operating system
 */
function getOS() {
	os = $.os;
	do {
		if (os.toLowerCase().indexOf("macintosh") >= 0) { 
			os = "mac";
			folderSeparator = "/";
			programFolder = folderSeparator + "Applications";
			BXSLT4InDesignAppl = tranformerName + ".app";
			break;
		}
		if (os.toLowerCase().indexOf("wind") >= 0) {
			os = "win";
			folderSeparator = "\\";
			programFolder = $.getenv("ProgramFiles");
			BXSLT4InDesignAppl = tranformerName + ".bat";
			break;
		}
		os = "unix";
		folderSeparator = "/";
		programFolder = folderSeparator + "Applications";
		BXSLT4InDesignAppl = tranformerName + ".sh";
	} while(false);
}


/****************************
 * hooks functions
 */
function getHookretval(settingsfilename) {
	if (!hooks[settingsFileName]) return 0;
	return 0;
}
function runHooks(callerscriptname, whichhook, args) {
	var hi;
	if (callerscriptname == "") {
		alert("Hooks callerscriptname is not set!! Hook call aborted!");
		return -1;
	}
	if (useHooks) {
		// export is DONE: check for hook to run after export
		if (endsWith(hooksBasePath,folderSeparator) == false) hooksBasePath += folderSeparator;
		//alert("hooksBasePath2: " + hooksBasePath);
		hooksPath = hooksBasePath + whichhook;
		hooksFolder = new Folder(hooksPath);
		//alert("hooksPath: " + hooksPath + "\n\nexists: " + hooksFolder.exists);
		if (hooksFolder.exists) {
			hooksScripts = hooksFolder.getFiles("*.jsx*");	// get .jsx and .jsxbin
			for (hi = 0; hi < hooksScripts.length; hi++) {
				//alert("caller: " + callerscriptname + "\nhook " + whichhook + "\nscript: " + hooksScripts[hi].fsName);
				if (startsWith(hooksScripts[hi].name,"__") == false) {	// skip those starting with two underscores
					hooks[settingsFileName] = {"callerscriptname":callerscriptname,"hook":whichhook,"args":args,"retval":0};
					app.doScript(hooksScripts[hi],ScriptLanguage.javascript);
					// do not remove params! we might want to retrieve the hook's retval or other props set by hook
				}
			}
		}
	}
	return 0;
}


/****************************
 * export a book
 */
function doBookExport() {
	var old_silentMode = silentMode;
	var old_BatchXSLTtriggerMode = BatchXSLTtriggerMode;
	var success = 0;
	doingBookExport = true;
	for (var b = 0; b < numBooksOpen; b++) {	// for each book file
		currentBook = app.books[b];
		// get book name
		currentBookName = currentBook.name;	// name of this book
		currentBookPath = currentBook.filePath;	// path to this book
		currentBookContent = currentBook.bookContents;	// contained documents

		// now open/export each document contained in book
		for (var d = 0; d < currentBookContent.length; d++) {	// for each document in a book
			var opendoc = openDocument(currentBookContent[d].fullName);
			// document is open: export it
			if (opendoc == error.NOERR) {
				if (d == 0) {	// the very first document of the book
					if (!(silentMode & 128)) silentMode = 8;	// no completion message for first document
					exportBookDocumentPDF = true;
				}
				else {
					if (!(silentMode & 128)) silentMode = 9;	// show main dialog for first document only
					exportBookDocumentPDF = false;
				}
				if (d < currentBookContent.length-1) BatchXSLTtriggerMode = 1;	// don't trigger BatchXSLT for all but last file
				else BatchXSLTtriggerMode = 0;	// trigger BatchXSLT on last file
				//alert("Doc #" + d + ": " + currentBookContent[d].fullName + "\nname: " + currentBookContent[d].name + "\ncurrentBookName: " + currentBookName + "\ncurrentBookPath: " + currentBookPath + "\nsilentMode: " + silentMode);
				success = initBXSLTExport(currentBookPath, currentBookName);
				// close document
				try { app.activeDocument.close(SaveOptions.no); } catch(e) {}
				if (success == error.EXPORT_CANCELLED) break;
			}
		}
		if (success == error.EXPORT_CANCELLED) break;
		// if not cancelled write book index File
		writeBookDocumentsXMLFile(currentBook);	// write a book index file
	}
	silentMode = old_silentMode;
	doingBookExport = false;
	exportBookDocumentPDF = false;	// make sure it is reset
	return(success);
}
function writeBookDocumentsXMLFile(book) {	// write an XML file with all book document names.
	if (book == null) return;
	build_export_paths(exportSubPath,"writeBookDocumentsXMLFile");	// need to know where to export
	var mefo_exists = create_folderstructure(mainExportFolderOut);
	if (mefo_exists != 0) {
		return(mefo_exists);	// retun last_errorCode = error.EXPORT_FOLDERSTRUCT_NOT_CREATED
	}
	// calc the relative path for the XSLCSS folder
	var xsl_relpath = "../XSLCSS/";
	if (xslcss_relpath_override != "") {
		xsl_relpath = xslcss_relpath_override; if (endsWith(xsl_relpath,folderSeparator) == false) xsl_relpath += folderSeparator;
	}
	else {
			if (exportSubPath != "") {
				var exportSubPath_arr = exportSubPath.split(folderSeparator);
				for (var i = 0; i < (exportSubPath_arr.length - 4); i++) {
					xsl_relpath = ".." + folderSeparator + xsl_relpath;
				}
			}
	}
	var outfolder = (new File(mainExportFolderOut)).fsName;
	var infolder = (new File(mainExportFolderIn)).fsName;
	var bookName = book.name;
	var bookXMLPathName = mainExportFolderOut + folderSeparator + bookName + ".xmi";
	var indexfile = new File(bookXMLPathName);
	indexfile.encoding = "UTF-8";
	var err = indexfile.open("w");
	var bookContent = book.bookContents;	// contained documents
	err = indexfile.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	//COMMENTED: DocBook not available      err = indexfile.write("<?xml-stylesheet href=\"" + xsl_relpath + "xml2DocBookX.xsl\" type=\"text/xsl\"?>\n");
	err = indexfile.write("<!DOCTYPE indbbookindex>\n<indbbookindex>\n");
	err = indexfile.write("<filename>" + bookName + ".xmi</filename>\n");
	err = indexfile.write("<filepath>" + outfolder + "</filepath>\n");

	err = indexfile.write("<book><path>" + book.filePath + "</path><name>" + book.name + "</name></book>\n");
	err = indexfile.write("<exportpath_in>" + infolder + "</exportpath_in>\n");
	err = indexfile.write("<exportpath_out>" + outfolder + "</exportpath_out>\n");
	//COMMENTED: DocBook not available      err = indexfile.write("<!-- this file, and the first file (idx=0) will be merged to the main DocBook named like the indesign book file -->\n");
	err = indexfile.write("<bookdocs>\n");
	for (var d = 0; d < bookContent.length; d++) {	// for each document in a book write out its name
		var docfile = new File(bookContent[d].fullName);
		var docbookxmlname = docfile.name.toString().substr(0,docfile.name.lastIndexOf(".")) + ".xml";
		var filePath = bookContent[d].filePath;	// may contain illegal and encoded characters
		var fullName = bookContent[d].fullName;
		var name = docfile.name;
		// clean names
		docbookxmlname = xPlatformName(docbookxmlname,totalencode);
		filePath = xPlatformName(filePath,totalencode);
		fullName = xPlatformName(fullName,totalencode);
		name = xPlatformName(name,totalencode);
		err = indexfile.write("<doc idx=\"" + bookContent[d].index + "\"><page>" + bookContent[d].documentPageRange + "</page><path>" + filePath + "</path><pathname>" + fullName + "</pathname><name>" + name + "</name><docbookxml>" + docbookxmlname + "</docbookxml></doc>\n");
	}
	err = indexfile.write("</bookdocs>\n</indbbookindex>\n");
	err = indexfile.close();

	//--------------------	
	// write the jobtickets to merge all book xml file into one single file and the transform to HTML
	newMergeJobTicketBatchXSLT(BXSLT4InDesignCommDir,bookXMLPathName);
}

function openDocument(docPathName) {
	var myDocument = null;
	var myDocumentWin = null;
	$.gc();
	var old_userInteractionLevel = app.scriptPreferences.userInteractionLevel;
	app.scriptPreferences.userInteractionLevel = UserInteractionLevels.neverInteract;
	try {
		myDocument = app.open(docPathName,false);	// do not show window now! this prevents some dialogs from beeing shown!
	}
	catch (Exception) {} 
	finally {
		if (myDocument == null) { 
			return(error.OPENENIG_DOC_FAILURE);
		}
		myDocumentWin = myDocument.windows.add();		// show the document window ...
		//var bounds = new Array(100,100,900,750);		// ... and size it (this also updates the screen)
		//myDocumentWin.bounds = bounds;
		//myDocumentWin.bringToFront();		// do not disturb user working while export is running: do not bring to front
		$.gc();
		$.sleep(500);	// let window breath
	}
	app.scriptPreferences.userInteractionLevel = old_userInteractionLevel;
	return(error.NOERR);	// document is open
}


function initBXSLTExport(the_docpath, the_docname) {

	// run with CS5 7.x.x, CS6 8.x.x, CC 9.x.x, CC 2014 10.x.x
	var versParts = appVersion.split(".");
	majorINDVersion = parseInt(versParts[0],10);
	//alert(majorINDVersion);
	if ((majorINDVersion >= 6) && (majorINDVersion <= 7)) is_CS4_or_later = true;
	else is_CS4_or_later = false;
	if ((majorINDVersion >= 5) && (majorINDVersion >= 7)) is_CS3_or_later = true;
	else is_CS3_or_later = false;
	if (majorINDVersion < 7) {
		//alert("This export script may not run with this version of InDesign!\nIt is designed to work with InDesign CS5 !\n\nProcessing aborted.");
		alert(loc(0));
		last_errorCode = error.WRONG_INDD_VERSION;
		return(error.WRONG_INDD_VERSION);
	}

	if (majorINDVersion < 5) splittingStoriesOption = 0;	// no splitting for CS2

	if (inxExportFormatString == "") {
		var ef = ExportFormat;
		var exportFormatIdentifier = inxExportFormatStringCS3;
		var props = ExportFormat.reflect.properties; 
		while(true) {
			for (var i = 0; i < props.length; i++) { 
				//$.writeln("this property " + props[i].name + " is " + ef[props[i].name]); 
				var propname = props[i].name;
				//alert(propname);
				if (propname.toUpperCase().indexOf(exportFormatIdentifier) >= 0) {
					inxExportFormatString = ef[props[i].name];
					break;
				}
			}
			if ((inxExportFormatString == "") && (exportFormatIdentifier != inxExportFormatStringCS5)) {
				// retry for CS5 InDesign Markup
				exportFormatIdentifier = inxExportFormatStringCS5;
				continue;
			}
			break;
		}
	}
	//alert(inxExportFormatString);
	if (inxExportFormatString == "") {
		//alert("The name of the 'InDesign IDML' Format could not be detected!\nPlease, edit the 'inxExportFormatString' variable in this script and try again!");
		alert(loc(1));
		last_errorCode = error.INXFORMAT_NOT_DETECTABLE;
		return(error.INXFORMAT_NOT_DETECTABLE);
	}

	// check if tranformer application may be found
	if (doStartBatchXSLT == true) {
		var bxsltappl_found = findBXSLTappl();
		if (bxsltappl_found == false) return;
	}


	// get available custom transforms
	availableOutputXSLs = get_xsl_files();
	if (availableOutputXSLs == null) {	// no output xsls found
		last_errorCode = error.MASTER_OUTPUTXSL_NOT_FOUND;
		return(error.MASTER_OUTPUTXSL_NOT_FOUND);
	}
	if (outputModeCustomXSL == "") {
		if (outputMode < availableOutputXSLs.length) {
			outputModeCustomXSL = availableOutputXSLs[outputMode];
		}
	}
	//alert("outputMode: " + outputMode + "\noutputModeCustomXSL: " + outputModeCustomXSL );



	if ((BatchXSLTtriggerMode & 2) == 2) {	// triggermode flag 2 set = trigger Transformer only
		//alert("mainExportFolder: "+ mainExportFolder + "\njtSubFolder: "+ jtSubFolder + "\nexportSubPath: "+ exportSubPath + "\nmainExportFolderIn: "+ mainExportFolderIn + "\nmainExportFolderOut: "+ mainExportFolderOut);
		build_export_paths(exportSubPath,"initBXSLTExport TRIGGER BXSLT",suppress_exportSubPath);
		//alert("BBBmainExportFolder: "+ mainExportFolder + "\njtSubFolder: "+ jtSubFolder + "\nexportSubPath: "+ exportSubPath + "\nmainExportFolderIn: "+ mainExportFolderIn + "\nmainExportFolderOut: "+ mainExportFolderOut);
		var success = callBXSLTExport(mainExportFolderIn);
		return(success);
	}


	// if no document is open, inform user to open one
	if (app.documents.length <= 0) {
		//alert("Please, open an InDesign document to export and re-call this script!");
		if ( (silentMode >= 0) && !(silentMode & 2) ) alert(loc(5));
		last_errorCode = error.NO_OPEN_DOC;
		return(error.NO_OPEN_DOC);
	}

	// check for open document
	var currentDocument = app.activeDocument,
		currentDocumentName = "",
		currentDocumentPath = "";
	if (typeof(the_docname) == 'undefined') {	// take from current open document
		currentDocumentName = currentDocument.name;
		currentDocumentPath = "";
		try { currentDocumentPath = currentDocument.filePath.toString(); } catch(e){}	// unsafed documents have no path
		if (currentDocumentFilePathName != "") {
			var tdf = new File(currentDocumentFilePathName);
			currentDocumentPath = tdf.path;
		}
		else {
			try {
				currentDocumentPath = currentDocument.filePath.toString();
			} catch (e) {
			}
		}
	}
	else {	// controlled from a book
		currentDocumentName = the_docname;
		currentDocumentPath = the_docpath;
	}

	retrieve_issue_year_date(currentDocumentName,currentDocumentPath);
	//alert("initBXSLTExport:retrieve_issue_year_date()\nfileDateFormat: " + fileDateFormat + "\n\nissueYear: " + issueYear + "\nissueDate: " + issueDate);
	objectShortcut = retrieve_objectShortcut(currentDocumentName);

	// check if document was saved, but don't, if we are in batch mode and evtl. opened a doc which was converted and therefore is unsaved
	// currentDocumentFilePathName is set when in batch mode.
	if ((currentDocumentFilePathName == "") && app.activeDocument.modified) {
		if ( !haveCallerScript && !doingBookExport && (silentMode >= 0) && !(silentMode & 2) ) {
			//alert("Please, save the document and then re-call this script!");
			//alert(loc(6));
			var stop = showSaveDialog();
			if (stop) {	// pressed OK button to save document first
				last_errorCode = error.UNSAVED_DOC;
				return(error.UNSAVED_DOC);
			}
		}
		//cancelled means: do export
	}

	// before initialize main dialog: run hook
	runHooks(scriptName,"beforeInitMainDialog", null);
	if (getHookretval(settingsFileName) != -1) {	// show main dialog if not -1 is returned
		// show main dialog
		if (!(silentMode & 1)) {
			var reloadedDialog = false;
			while (true) {
				var goahead = 0;	// 0 = cancelled dialog
										// 1 = do export
										// 2 = load new settings
										// 3 = set factory settings
				if (reloadedDialog) runHooks(scriptName,"beforeInitMainDialog", null);	// re hook after reloading main dialog
				goahead = showMainDialog();
				switch (goahead) {
					case 0: return(error.EXPORT_CANCELLED);	// cancelled
					case 2:	// load settings file
						break;
					case 3:	// save settings
						break;
					case 4:	// set factory settings
						break;
					default: break;	// go, do export
				}
				if ((goahead == 2) || (goahead == 3) || (goahead == 4)) {
					reloadedDialog = true;
					continue;	// revert or load default settings
				}
				var fieldsCheck = checkDialogFields();
				if (fieldsCheck != 0) {
					reloadedDialog = true;
					continue;
				}
				break;
			}
		}
	}

	if (callerScript.toLowerCase().indexOf("batch") >= 0) {	// Batch mode: state the path only
		build_export_paths(exportSubPath,callerScript,suppress_exportSubPath);
	}
	else build_export_paths("","initBXSLTExport (2)",suppress_exportSubPath);

	// check if Export folder structure may be found
	var mefi_exists = create_folderstructure(mainExportFolderIn);
	if (mefi_exists != 0) {
		return(mefi_exists);	// retun last_errorCode = error.EXPORT_FOLDERSTRUCT_NOT_CREATED
	}

	// start exporting the document
	var created = callBXSLTExport(mainExportFolderIn);

	if (allowRevertDocument && !closeDocumentAfterDone && (splittingStoriesOption > 0) && (numSplittedStoryBoxes > 0)) {
		//message("Due to the requested story splitting, the document's text frame chains probably are split. Re-open the document or allow the following message to revert to the last saved version!");
		//message(loc(8));
		var safe_userInteractionLevel = app.scriptPreferences.userInteractionLevel;
		app.scriptPreferences.userInteractionLevel = UserInteractionLevels.neverInteract;	// do not show the revert dialog
		app.activeDocument.revert(true);
		app.scriptPreferences.userInteractionLevel = safe_userInteractionLevel;
	}

	if (created == error.NOERR) {
		if ( (silentMode >= 0) && !(silentMode & 8) && (showMessage == true) ) {
			var f = new File(expectedXMLfilePath);
			var baseExportFolder = new Folder(mainExportFolderIn);
			//var mess = "Base Export for '" + applicationHumanName + "' to Folder:\n'" + baseExportFolder.fsName + "'\ncomplete.";
			var mess = loc(9,baseExportFolder.fsName);
			if (BXSLTstarted == true) {
				//mess += "\n\n'" + applicationHumanName + "' has been started to create ePaper.\n\nSwitch to '" + applicationHumanName + "' and await completion.\n\nYou then may open the file:\n'" + f.fsName + "'\nin your XML-aware browser.\n\n\nClick the 'OK' button to close this message";
				mess += loc(10,f.path);
				if (closeDocumentAfterDone) mess += loc(11);
			}
			alert(mess);
		}
	}
	else {
		//alert("Document export to folder\n" + mainExportFolder + "\n\nfailed.");
		if ( (silentMode >= 0) && !(silentMode & 2) ) alert(loc(12,mainExportFolder,created));
	}

	if (showExportFolder && !doingBookExport) {
		var aliassource = new File (mainExportFolder);
		var aliasfile = new File ("~/Desktop/Export Alias");
		//alert(aliassource.fsName+"\nexists: " + aliassource.exists + "\naliasfile: " + aliasfile.fsName + "\naliasfile exists: " + aliasfile.exists);
		if (aliasfile.exists == false) {
			var aliascreated = aliasfile.createAlias(aliassource,false);
			//alert("sourceError: "+aliassource.error+"\naliasError: "+aliasfile.error);
		}
		var folderopen = aliasfile.execute();
	}

	if (closeDocumentAfterDone) app.activeDocument.close(SaveOptions.no);
	return (created);
}


/****************************
 * here we go.....
 */
function callBXSLTExport(exportFolder) {
	if ( (BatchXSLTtriggerMode == 2) && ((exportFolder == null) || (exportFolder == "")) ) {	// start tranformer only
		BXSLTstarted = BXSLTstart();
		if (BXSLTstarted == false) {
			//alert(applicationHumanName + " could not be started!\nStart it manually!!");
			if ( (silentMode >= 0) && !(silentMode & 2) ) alert(loc(13));
			last_errorCode = error.COULD_NOT_START_BATCHXSLT;
			return(error.COULD_NOT_START_BATCHXSLT);
		}
		return(error.NO_ERR);
	}

	if ( (exportFolder == null) || (exportFolder == "") ) {
		last_errorCode = error.NO_EXPORT_FOLDER;
		return(error.NO_EXPORT_FOLDER);
	}
	var baseExportFolder = new Folder(exportFolder);
	
	// evtl. do document export
	if ((BatchXSLTtriggerMode & 2) != 2) {	// triggermode flag 2 set = trigger only
		// change some settings
		var scrpref = app.scriptPreferences;
		//scrpref.userInteractionLevel = UserInteractionLevels.neverInteract;
		var old_display_settings = app.displayPerformancePreferences.defaultDisplaySettings
		app.displayPerformancePreferences.defaultDisplaySettings = ViewDisplaySettings.highQuality;
	
		// let's do the export
		doDocumentExport(exportFolder);
	
		// restore settings
		app.displayPerformancePreferences.defaultDisplaySettings = old_display_settings;
		scrpref.userInteractionLevel = UserInteractionLevels.interactWithAll;
		$.gc();
	}

	// we are done... evtl. create a file to trigger Transformer
	BXSLTstarted = false;
	if ((BatchXSLTtriggerMode & 1) != 1) {	// triggermode flag 1 set = no trigger now
		triggerBatchXSLT(baseExportFolder);
	
		if (doControlBatchXSLT == true) {
			// evtl. create a JobTicket file for transformer
			newJobTicketBatchXSLT(BXSLT4InDesignCommDir);
		}
		// start Transformer
		if (doStartBatchXSLT == true) {
			BXSLTstarted = BXSLTstart();
			if (BXSLTstarted == false) {
				//alert(applicationHumanName + " could not be started!\nStart it manually!!");
				if ( (silentMode >= 0) && !(silentMode & 2) ) alert(loc(13));
				last_errorCode = error.COULD_NOT_START_BATCHXSLT;
				return(error.COULD_NOT_START_BATCHXSLT);
			}
		}
	}

	last_errorCode = error.NOERR;
	return(error.NOERR);
}

function BXSLTstart() {	// start the Transformer
	if (BXSLT4InDesignPath == "") return(false);
	if (doStartBatchXSLT != true) return(false);
	var app2start = new File (BXSLT4InDesignPath);
	var started = app2start.execute();
	return(started);
}




/****************************
 * Export the document to IDML, PDF and JPEG
 */
function doDocumentExport(theExportFolder) {
	var theFilePath, aFile,
		theDocument = "", theDocumentName = "",
		pageStr, theFilePathBase,
		ctr;
	theDocument = app.activeDocument;
	theDocumentName = theDocument.name;
	theBaseDocumentName = theDocumentName;

	// get the document name without file extension
	if (theBaseDocumentName.indexOf(".indd") > 0) {
		theBaseDocumentName = theBaseDocumentName.substr(0,theDocumentName.lastIndexOf(".indd"));
	}
	// replace unwanted chars in filename:  like #*/
	//theBaseDocumentName = theBaseDocumentName.replace(/\//,"s");
	if (totalencode) theBaseDocumentName = xPlatformName(theBaseDocumentName,true);
	else theBaseDocumentName = xPlatformName(theBaseDocumentName,false);
	while (theBaseDocumentName.charAt(theBaseDocumentName.length - 1) == ".") {
		theBaseDocumentName = theBaseDocumentName.substr(0,theBaseDocumentName.length - 1);
	}

	// before document export: check for hook to run
	runHooks(scriptName,"beforeDocExport", null);
	if (getHookretval(settingsFileName) == -1) return;	// do not export if returned -1


	// Export JPEG page per page
	if (exportPageJPEGs) {
		for (ctr = 0; ctr < theDocument.pages.length; ctr++) {
			if (!(silentMode & 4)) {
				var mess = loc(17,(ctr+1)) + "     .";
				message(mess);
			}
			$.sleep(100);

			var sectionName = theDocument.pages.item(ctr).appliedSection.name,
				sectionPrefix = theDocument.pages.item(ctr).appliedSection.sectionPrefix,
				includeSectionPrefix = theDocument.pages.item(ctr).appliedSection.includeSectionPrefix,
				pageName = theDocument.pages.item(ctr).name;
			if (includeSectionPrefix == false) {	// section prefix is already in page number:add it
				pageName = sectionPrefix + pageName;
			}
			// The name of the exported files will be: documentName + pageStr + ".jpg"
			pageStr = "_p" + (ctr + 1);	// the page number string
			//if (ctr == 0) pageStr = "";		// none for page #1
			theFilePathBase = theExportFolder + folderSeparator + theBaseDocumentName + pageStr;

			// export page to JPG
			try { app.jpegExportPreferences.jpegExportRange = ExportRangeOrAllPages.exportRange; } catch (e) {}
			try { app.jpegExportPreferences.pageString = pageName; } catch (e) {}
			try { app.jpegExportPreferences.jpegColorSpace = JpegColorSpaceEnum.RGB; } catch (e) {}
			try { app.jpegExportPreferences.embedColorProfile = pageJPEGembedColorProfile; } catch (e) {}
			try { app.jpegExportPreferences.jpegQuality = pageImageQualityOriginal; } catch (e) {}
			try { app.jpegExportPreferences.exportingSelection = false; } catch (e) {}
			try { app.jpegExportPreferences.exportingSpread = false; } catch (e) {}
			try { app.jpegExportPreferences.jpegRenderingStyle = JPEGOptionsFormat.baselineEncoding; } catch (e) {} 	// or progressiveEncoding
			try {	// original must have at least same DPI
				if (parseInt(pageImageDPI,10) < pageImageDPIOriginal) app.jpegExportPreferences.exportResolution = pageImageDPIOriginal;
				else app.jpegExportPreferences.exportResolution = parseInt(pageImageDPI,10);
			} catch (e) {}
			theFilePath = theFilePathBase + ".jpg";
			aFile = new File(theFilePath);
			// before page JPEG export: check for hook to run
			runHooks(scriptName,"beforePageJPEGExport", { "theFilePath": theFilePath, "jpegExportPreferences": app.jpegExportPreferences });
			if (getHookretval(settingsFileName) != -1) {	// do write page JPEG if not -1 is returned
				try {
					theDocument.exportFile(ExportFormat.jpg, aFile, false);
				} catch (e) {}
			}
			// after page JPEG export: check for hook to run
			runHooks(scriptName,"afterPageJPEGExport", { "theFilePath": theFilePath });
		}
		messageClose();
	}


	// Export PDF page per page
	if (exportPagePDFs) {
		var usepagepdfpresets = null,
			isApplpfdpreferences = false;
		try { usepagepdfpresets = exportPDFpresets[exportPagePDFusePresetIndex]; } catch(e){}
		if (usepagepdfpresets == null) {
			isApplpfdpreferences = true;
			usepagepdfpresets = app.pdfExportPreferences;	// take the default ones
		}

		for (ctr = 0; ctr < theDocument.pages.length; ctr++) {
			if (!(silentMode & 4)) {
				if (isApplpfdpreferences)  message(loc(18,(ctr+1)) + "     .");
				else message(loc(18,(ctr+1)) + " (preset: " + exportPagePDFusePresetName + ")       .");
			}

			var sectionName = theDocument.pages.item(ctr).appliedSection.name,
				sectionPrefix = theDocument.pages.item(ctr).appliedSection.sectionPrefix,
				includeSectionPrefix = theDocument.pages.item(ctr).appliedSection.includeSectionPrefix,
				pageName = theDocument.pages.item(ctr).name;
			if (includeSectionPrefix == false) {	// section prefix is already in page number:add it
				pageName = sectionPrefix + pageName;
			}
			// The name of the exported files will be: documentName + pageStr + ".pdf"
			var pageStr = "_p" + (ctr + 1),	// the page number string
				theFilePathBase = theExportFolder + folderSeparator + theBaseDocumentName + pageStr;
			//alert("sectionName: " + sectionName + "\nsectionPrefix: " + sectionPrefix + "\nincludeSectionPrefix: " + includeSectionPrefix + "\npageName: " + pageName + "\ntheFilePathBase: " + theFilePathBase);

			// export to PDF
			$.sleep(100);
			app.pdfExportPreferences.pageRange = pageName;

			try { app.pdfExportPreferences.viewPDF = false; } catch (e) {}
			try { usepagepdfpresets.exportReaderSpreads = false; } catch (e) {}
			if ((isApplpfdpreferences == true) || (forcePDFinteractivity == true)) {
				try { usepagepdfpresets.optimizePDF = false; } catch (e) {}	// set to true only, if you are sure that the PDF export will not crash
																					// even when manually call PDF export, in certain cases, the Adobe's own PDF export may crash when set to 'optimize'
				// set here your PDF preferences if you want others than the current defaults
				try { usepagepdfpresets.includeHyperlinks = true; } catch (e) {}
				try { usepagepdfpresets.cropImagesToFrames = true; } catch (e) {}
				try { usepagepdfpresets.interactiveElementsOption = InteractiveElementsOptions.APPEARANCE_ONLY; } catch (e) {}
				try { usepagepdfpresets.includeBookmarks = true; } catch (e) {}
				try { usepagepdfpresets.includeICCProfiles = true; } catch (e) {}
			}

			theFilePath = theFilePathBase + ".pdf";
			aFile = new File(theFilePath);
			// before page PDF export: check for hook to run
			runHooks(scriptName,"beforePagePDFExport", { "theFilePath": theFilePath, "usepagepdfpresets": usepagepdfpresets });
			if (getHookretval(settingsFileName) != -1) {	// do write page JPEG if not -1 is returned
				try {
					if (isApplpfdpreferences == true) theDocument.exportFile(ExportFormat.pdfType, aFile, false);
					else theDocument.exportFile(ExportFormat.pdfType, aFile, false, usepagepdfpresets);
				} catch (e) {}
			}
			// after page PDF export: check for hook to run
			runHooks(scriptName,"afterPagePDFExport", { "theFilePath": theFilePath });
		}
		messageClose();
	}


	// export the whole document to a PDF file
	// if we do a book export, the entire document is exported in function doBookExport()
	if ((!doingBookExport || exportBookDocumentPDF) && (exportDocumentPDF == true)) {
		var usedocumentpdfpresets = null,
			isApplpfdpreferences = false;
		try { usedocumentpdfpresets = exportPDFpresets[exportDocumentPDFusePresetIndex]; } catch(e){}
		if (usedocumentpdfpresets == null) {
			isApplpfdpreferences = true;
			usedocumentpdfpresets = app.pdfExportPreferences;	// take the default ones
		}

		var myBaseDocumentName = theBaseDocumentName;
		if (exportBookDocumentPDF) {	// get the book name without file extension
			myBaseDocumentName = currentBookName;
			// replace unwanted chars in filename:  like #*/
			//myBaseDocumentName = myBaseDocumentName.replace(/\//,"s");
			if (totalencode) myBaseDocumentName = xPlatformName(myBaseDocumentName,true);
			else myBaseDocumentName = xPlatformName(myBaseDocumentName,false);
			while (myBaseDocumentName.charAt(myBaseDocumentName.length - 1) == ".") {
				myBaseDocumentName = myBaseDocumentName.substr(0,myBaseDocumentName.length - 1);
			}
		}
		var theFilePathBase = theExportFolder + folderSeparator + myBaseDocumentName;
		theFilePath = theFilePathBase + "_doc.pdf";
		thePDFdocname = myBaseDocumentName + "_doc.pdf";
		//message("Exporting document PDF to '" + thePDFdocname + "' ...");
		if (!(silentMode & 4)) {
			if (isApplpfdpreferences) message(loc(14,thePDFdocname));
			else message(loc(14,thePDFdocname) + " (preset: " + exportDocumentPDFusePresetName + ")");
		}

		app.pdfExportPreferences.pageRange = PageRange.allPages;
		try { app.pdfExportPreferences.viewPDF = false; } catch (e) {}
		if (exportDocumentPDFspreads == true) { try { usedocumentpdfpresets.exportReaderSpreads = true; } catch (e) {} }
		else { try { usedocumentpdfpresets.exportReaderSpreads = false; } catch (e) {} }
		if ((isApplpfdpreferences == true) || (forcePDFinteractivity == true)) {
			// set here your PDF preferences if you want others than the current defaults
			try { usedocumentpdfpresets.optimizePDF = false; } catch (e) {}	// set to true only, if you are sure that the PDF export will not crash
			try { usedocumentpdfpresets.includeHyperlinks = true; } catch (e) {}
			try { usedocumentpdfpresets.cropImagesToFrames = true; } catch (e) {}
			try { usepagepdfpresets.interactiveElementsOption = InteractiveElementsOptions.APPEARANCE_ONLY; } catch (e) {}
			try { usedocumentpdfpresets.includeBookmarks = true; } catch (e) {}
			try { usedocumentpdfpresets.includeICCProfiles = true; } catch (e) {}
		}

		aFile = new File(theFilePath);
		// before document PDF export: check for hook to run
		runHooks(scriptName,"beforeDocPDFExport", { "theFilePath": theFilePath, "usedocumentpdfpresets": usedocumentpdfpresets });
		if (getHookretval(settingsFileName) != -1) {	// do write document PDF if not -1 is returned
			try {
				if (createFakeDocumentPDF == false) {
					if (!exportBookDocumentPDF) {
						if (isApplpfdpreferences == true) theDocument.exportFile(ExportFormat.PDF_TYPE, aFile, false);
						else theDocument.exportFile(ExportFormat.PDF_TYPE, aFile, false, usedocumentpdfpresets);
					}
					else {	// write a PDF from entire book
						try {
						if (isApplpfdpreferences == true) currentBook.exportFile (ExportFormat.PDF_TYPE, aFile, false, null, currentBookContent, "", false);
						else currentBook.exportFile (ExportFormat.PDF_TYPE, aFile, false, usedocumentpdfpresets, currentBookContent, "", false);
						} catch(err) {
						}
					}
				}
				else {	// create fake zero bytes file
					aFile.open("w"); aFile.close();
				}
			} catch (e) {}
		}
		// after document PDF export: check for hook to run
		runHooks(scriptName,"afterDocPDFExport", { "theFilePath": theFilePath });
		messageClose();
	}


	// evtl split story frames before exporting
	if (is_CS3_or_later && (splittingStoriesOption > 0)) {
		if (!(silentMode & 4)) {
			messageClose();
			message(loc(70));
		}
		// before story splitting: check for hook to run
		runHooks(scriptName,"beforeSplitStories", null);
		if (getHookretval(settingsFileName) != -1) {	// do write page JPEG if not -1 is returned
			switch(splittingStoriesOption) {
				case 1:
					var scriptFile = new File(getScriptsPath() + folderSeparator + utilitiesFilePath + folderSeparator + "SplitStoryChainsPerPage.jsx");
					app.scriptArgs.setValue("splitAllTextFrames", "0");
					app.doScript(scriptFile,ScriptLanguage.javascript);
					numSplittedStoryBoxes = parseInt(app.scriptArgs.getValue("totalSplittedTextFrames"));
					break;
				case 2:
					var scriptFile = new File(getScriptsPath() + folderSeparator + utilitiesFilePath + folderSeparator + "SplitStoryChainsPerPage.jsx");
					app.scriptArgs.setValue("splitAllTextFrames", "1");
					app.doScript(scriptFile,ScriptLanguage.javascript);
					numSplittedStoryBoxes = parseInt(app.scriptArgs.getValue("totalSplittedTextFrames"));
					break;
			}
			if (!(silentMode & 4)) message(loc(71));	// let the document breath
			$.sleep(app.activeDocument.pages.length * 20);
		}
		// after story splitting: check for hook to run
		runHooks(scriptName,"afterSplitStories", null);
		messageClose();
	}

	// make all invisible conditional text visible
	// this MUST happen AFTER stories splitting
	// before page switch on text conditions: check for hook to run
	runHooks(scriptName,"beforeTextCondVisible", null);
	if (getHookretval(settingsFileName) != -1) {	// do turn on all conditional text if not -1 is returned
		if (showAllConditionalText && (majorINDVersion >= 6)) {	// CS4 and newer only
			textConditionsVisible();
		}
	}
	// after page switch on text conditions: check for hook to run
	runHooks(scriptName,"afterTextCondVisible", null);


	// now, copy all images to export folder
	// this MUST happen AFTER stories splitting because anchored images will get new IDs
	// before exporting images: check for hook to run
	runHooks(scriptName,"beforeExportImages", null);
	if (getHookretval(settingsFileName) != -1) {	// export images if not -1 is returned
		//		  alert("exportImages:"+exportImages + "\nimageMissingCreateFromPagePreview:"+imageMissingCreateFromPagePreview)
		if (exportImages > 0) {
			switch (imageMissingCreateFromPagePreview) {
				case 0:	// always convert from original image
				case 1:	// extract images from screen page when original image file is missing (image box ectracted by InDesign script)
				  //  (available when ImageMagick is installed)
					copyImages(theExportFolder);
					break;
				case 2:	// always create from image box
				  // the only option without installed ImageMagick: InDesign exports the Images [this script]
					create_all_images_from_box(theExportFolder);
					break;
				case 3:	// we have to mark (label) the position of anchored images, for we can extract them from page preview
				case 4:	
					scanPagesForAnchoredGraphics();
					break;
			}
			messageClose();
		}
	}
	// after exporting images: check for hook to run
	runHooks(scriptName,"afterExportImages", null);

	// export the whole document to a single IDML file
	// before exporting IDML: check for hook to run
	runHooks(scriptName,"beforeIDMLExport", null);
	if (getHookretval(settingsFileName) != -1) {	// export images if not -1 is returned
		if (exportINX > 0) {
			if (!(silentMode & 4)) {
				messageClose();
				var format = "IDML";
				if (majorINDVersion < 7) format = "INX";
				message(loc(15,format));	// Exporting Document to IDML Format ...
			}
			theFilePath = theExportFolder + "/" + theBaseDocumentName;
			if (majorINDVersion >= 7) theFilePath += ".idml";
			else theFilePath += ".inx";
			expectedXMLfilePath = mainExportFolderOut + "/" + theBaseDocumentName + ".xml";
			aFile = new File(theFilePath);
			theDocument.exportFile(inxExportFormatString, aFile, true);
			messageClose();
		}
	}
	// after exporting IDML: check for hook to run
	runHooks(scriptName,"afterIDMLExport", null);

	messageClose();

	// after document export: check for hook to run after export
	runHooks(scriptName,"afterDocExport", null);
	return;

}



/****************************
 * create a trigger file to start transformation
 */
function triggerBatchXSLT(atFolder) {
	if (BXSLTTriggerFile == "") return;
	var triggerFile = new File(atFolder.fsName + folderSeparator + BXSLTTriggerFile);
	var err = triggerFile.open("w");
	err = triggerFile.write(applicationHumanName + " Trigger File");
	err = triggerFile.close();
}



/****************************
 * create a jobticket for the transformer
 */
function newJobTicketBatchXSLT(path) {
	if (path == null || path == "") return;
	var f = new File(path);
	//alert("newJobTicketBatchXSLT()\npath: " + path +"\nf.fsName: " + f.fsName +"\nexists: " + f.exists);
	if (f.exists == false) {
		var fldr = new Folder(path);
		fldr.create();
		f = new File(path);
		if (f.exists == false) {
			alert(loc(25));
			return;
		}
	}

	// -----------------------------------------
	// run hooks before write JobTickets
	runHooks(scriptName,"beforeWriteJobTickets", { "commpath":path });
	if (getHookretval(settingsFileName) == -1) return;	// do not write jobTickets if -1 is returned

	// get the document name without file extension
	var theDocumentName = "";
	if (!doingBookExport && (app.documents.length > 0)) {	// this is for manual exports from menu
		theDocumentName = app.activeDocument.name;
		if (theDocumentName.indexOf(".indd") > 0) theDocumentName = theDocumentName.substr(0,theDocumentName.lastIndexOf(".indd"));
	}

	// calc the relative path for the XSLCSS folder
	var xsl_relpath = "../XSLCSS/";
	var css_relpath = "";
	if (xslcss_relpath_override != "") {
		xsl_relpath = xslcss_relpath_override; if (endsWith(xsl_relpath,folderSeparator) == false) xsl_relpath += folderSeparator;
		css_relpath = xslcss_relpath_override; if (endsWith(css_relpath,folderSeparator) == false) css_relpath += folderSeparator;
	}
	else {
		if (startsWith(cssName,("\." + folderSeparator)) == true) {	// place css in output folder
			cssName = cssName.substr(2);
		}
		else {
			css_relpath = "../XSLCSS/";
			if (exportSubPath != "") {
				var exportSubPath_arr = exportSubPath.split(folderSeparator);
				for (var i = 0; i < (exportSubPath_arr.length - 4); i++) {
					xsl_relpath = "../" + xsl_relpath;
					css_relpath = "../" + css_relpath;
				}
			}
		}
	}
	//alert("WWmainExportFolder: "+ mainExportFolder + "\njtSubFolder: "+ jtSubFolder + "\nmainExportFolderIn: "+ mainExportFolderIn + "\nmainExportFolderOut: "+ mainExportFolderOut);


	// write the first jobticket to transform to epaper xml
	var jtname = new Date().getTime() + "override.jt";
	var commpath = path;
	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	var jtFile = new File(commpath + jtname);
	jtFile.encoding = "UTF-8";
	var err = jtFile.open("w");

	err = jtFile.write("init_previous_jobticket = 0\n");
	err = jtFile.write("jobticketVersion = 10.0\n");
	err = jtFile.write("applicationPurpose = InDesign_IMX_Transformer\n");
	err = jtFile.write("mode = 1\n");
	err = jtFile.write("\n");
	err = jtFile.write("jt_triggerfile = " + BXSLTTriggerFile + "\n");
	if (callerScript.toLowerCase().indexOf("batch") >= 0) {	// Batch mode: state the path only
		err = jtFile.write("sourcePathName = " + pathAdapter(mainExportFolderIn) + folderSeparator + "\n");
	}
	else {
		if (theDocumentName != "") err = jtFile.write("sourcePathName = " + pathAdapter(mainExportFolderIn) + folderSeparator + theBaseDocumentName + ".idml\n");
		else err = jtFile.write("sourcePathName = " + pathAdapter(mainExportFolderIn) + folderSeparator + "\n");
	}
	if (majorINDVersion >= 7) err = jtFile.write("xslPathName = XSL/IMXepaper.xsl\n");
	else err = jtFile.write("xslPathName = XSL/INXepaper.xse\n");
	err = jtFile.write("outputPathName = " + pathAdapter(mainExportFolderOut) + folderSeparator + "\n");
	err = jtFile.write("externalProcessTimeout = " + externalProcessTimeout + "\n");
	err = jtFile.write("continueXMLparseOnFatalError = " + continueXMLparseOnFatalError + "\n");


	//if (outputMode > 3 ) err = jtFile.write("tp_3 = outputMode=0\n");
	//else err = jtFile.write("tp_3 = outputMode=" + outputMode + "\n");
	err = jtFile.write("tp_3 = outputMode=" + outputMode + "\n");
	if (pageImageWidth != "") err = jtFile.write("tp_5 = pageJPEGScale=" + pageImageWidth + "\n");
	if (pageImageQuality != "") err = jtFile.write("tp_7 = pageJPEGQuality=" + pageImageQuality + "\n");
	if (pageImageDPI != "") err = jtFile.write("tp_9 = pageJPEGdpi=" + pageImageDPI + "\n");
	err = jtFile.write("tp_13 = imageTYPE=" + exportImages + "\n");

	if (imageParameters != "") {
		var imgparmsArr = imageParameters.split("//");
		var imgparms = imgparmsArr[0].split(";");
		err = jtFile.write("tp_17 = imageSCALE=" + imgparms[0] + "\n");
		if (imgparms.length >= 2) {
			err = jtFile.write("tp_21 = imagePARAMS=" + imgparms[1] + "\n");
		}
		if (imgparms.length >= 3) {
			if (imgparms[2] == 'nocrop') err = jtFile.write("tp_67 = imageCROP=0\n");
			else err = jtFile.write("tp_67 = imageCROP=1\n");
		}
		if (imgparmsArr.length > 1) jtFile.write("tp_69 = imagesMORE=" + imgparmsArr[1] + "\n");
		else jtFile.write("tp_69 = imagesMORE=\n");
	}
	err = jtFile.write("tp_23 = catchRadius=" + catchRadius + "\n");
	err = jtFile.write("tp_25 = allBoxesNoCatch=" + allBoxesNoCatch + "\n");
	if (chainedBoxesNoCatch == true) err = jtFile.write("tp_27 = chainedBoxesNoCatch=1\n");
	else  err = jtFile.write("tp_27 = chainedBoxesNoCatch=0\n");
	if (unassignedBoxesNoCatch == true) err = jtFile.write("tp_90 = unassignedBoxesNoCatch=1\n");
	else  err = jtFile.write("tp_90 = unassignedBoxesNoCatch=0\n");
	if (emptyBoxesNoCatch == true) err = jtFile.write("tp_91 = emptyBoxesNoCatch=1\n");
	else  err = jtFile.write("tp_91 = emptyBoxesNoCatch=0\n");
	if (imageBoxesNoCatch == true) err = jtFile.write("tp_114 = imageBoxesNoCatch=1\n");
	else  err = jtFile.write("tp_114 = imageBoxesNoCatch=0\n");
	if (textBoxesNoCatch == true) err = jtFile.write("tp_120 = textBoxesNoCatch=1\n");
	else  err = jtFile.write("tp_120 = textBoxesNoCatch=0\n");
	err = jtFile.write("tp_33 = XSLCSSPath=" + xsl_relpath + "\n");
	err = jtFile.write("tp_35 = CSSpath=" + css_relpath + "\n");
	err = jtFile.write("tp_37 = CSSname=" + cssName + "\n");
	err = jtFile.write("tp_43 = FOLDERINDEXNAME=\n");
//	err = jtFile.write("tp_45 = INTERMEDIARY_XML_preserve=1\n");
//	err = jtFile.write("tp_46 = DEBUG=1\n");
//	err = jtFile.write("tp_47 = DEBUGIMAGES=1\n");
//	err = jtFile.write("tp_48 = DEBUGPAGEIMAGES=1\n");
	err = jtFile.write("tp_49 = documentPDFname=" + thePDFdocname + "\n");
	err = jtFile.write("tp_51 = TABLE_CELLS_WIDTH_PRESERVE=" + TABLE_CELLS_WIDTH_PRESERVE + "\n");
	err = jtFile.write("tp_53 = TABLE_BORDER_COLLAPSE=" + TABLE_BORDER_COLLAPSE + "\n");
	err = jtFile.write("tp_71 = TABLE_AS_BLOCK=" + TABLE_AS_BLOCK + "\n");
	err = jtFile.write("tp_55 = imageCopyToOutput=" + copyOriginalImageFiles + "\n");
	err = jtFile.write("tp_59 = fontsizeMinimum=" + fontsizeMinimum + "\n");
	switch(magnifyingGlass) {	// maginifying Glass option
		case 0: err = jtFile.write("tp_61 = magnifyingGlass=0\n"); break;	// do not enable, allow normal mouse over only
		case 1: err = jtFile.write("tp_61 = magnifyingGlass=10\n"); break;	// enable glass and set default view mode with floater and allow toggle
		case 2: err = jtFile.write("tp_61 = magnifyingGlass=111\n"); break;	// enable glass and set default view mode with magnifying glass and allow toggle
		case 3: err = jtFile.write("tp_61 = magnifyingGlass=110\n"); break;	// enable glass and set view mode with magnifying glass and DO NOT allow toggle
	}
	jtFile.write("tp_63 = outputModeCustomXSL=" + outputModeCustomXSL + "\n");	// set XSL name to override output XSL to dedicated XSL
	jtFile.write("tp_73 = imageOutputFinishParam=" + imageOutputFinishParam + "\n");	// strip all comments and profiles from output image
	jtFile.write("tp_74 = imagePreviewUseOriginals=1\n");	// 1 = do not convert images created from previews - just copy and link it, 0 = convert them too -->
//	jtFile.write("tp_75 = CSSclassNamesNoUnderscore=1\n");	// 0 or empty for normal, 1 = css class names should not contain underscore chars (OpenOffice does not like underscore)

	if (exportPROFeatures == true) {
		jtFile.write("tp_96 = OUTPUT_FEATURES=2\n");
		jtFile.write("tp_97 = preconvertTextFlags=0\n");
		jtFile.write("tp_98 = includeMetaData=1\n");
		jtFile.write("tp_99 = CSSexpanded=1\n");
		jtFile.write("tp_100 = excludeNotes=0\n");
	}
	else {
		jtFile.write("tp_95 = CSSclassNamesNoUnderscore=0\n");
		jtFile.write("tp_96 = OUTPUT_FEATURES=0\n");
		jtFile.write("tp_97 = preconvertTextFlags=31\n");
		jtFile.write("tp_98 = includeMetaData=0\n");
		jtFile.write("tp_99 = CSSexpanded=0\n");
		jtFile.write("tp_100 = excludeNotes=1\n");
	}

//	jtFile.write("tp_102 = CROPBOXmode=0\n");	//  how to handle CropBox for PDFs (default = 124)
												//	0 = do not add a "-define pdf:use-cropbox=true" command
												//	    and do not try to replace the artbox in pdf files with cropbox
												//	    and do not add a -crop command to ImageMagick
												//	    means: leave PDF as is and export uncropped
												//	addable flags like "124"
												//	1 = try to replace the 'ArtBox' in pdf files with 'CropBox' if 'CropBox' is not already set
												//	2 = add a "-define pdf:use-cropbox=true" command if PDF has 'CropBox'
												//	4 = add a "-crop WxHxXxY" command
	jtFile.write("tp_103 = fontsizeFactor=" + fontsizeFactor + "\n");
	jtFile.write("tp_104 = paragraphAttribsSuppress=1\n");	// adable flags: suppress paragraph attributes like line-height (leading) in the HTML div/@style attribute
															// 0 = expoprt all attribs
															// 1 = suppress leading
															// -1 = supress all manually set style changes = dont create a style attribute

	jtFile.write("tp_105 = characterAttribsSuppress=0\n");	// adable flags: suppress manually set character attributes like font-size in the HTML span/@style attribute
															// 0 = leave attribs as is
															// 1 = suppress manually set font-family
															// 2 = suppress manually set font-size
															// -1 = supress all manually set style changes = dont create a style attribute
	jtFile.write("tp_106 = includeXMLtags=" + includeXMLtags + "\n");

	jtFile.write("tp_107 = groupBoxesToArticles=" + groupBoxesToArticles + "\n");
	jtFile.write("tp_108 = sortBoxesByAreaSize=" + sortBoxesByAreaSize + "\n");
	jtFile.write("tp_109 = sortBoxesByYposition=" + sortBoxesByYposition + "\n");
	jtFile.write("tp_110 = sortBoxYtolerance=" + sortBoxYtolerance + "\n");
	jtFile.write("tp_111 = imageDoROTATE=" + imageDoROTATE + "\n");
	jtFile.write("tp_65 = imageEXCLUDE=" + imageEXCLUDE + "\n");
	
	if (totalencode == true) jtFile.write("tp_112 = XPLATFORM_NAMES=29\n");	// defaults to 2: tell the tranformer to encode names
	else jtFile.write("tp_112 = XPLATFORM_NAMES=0\n");	// defaults to 2: tell the tranformer to encode names
	jtFile.write("tp_113 = excludeHiddenLayers=" + excludeHiddenLayers + "\n");

	err = jtFile.write("tp_115 = suppressAllEmptyBox=" + suppressAllEmptyBox + "\n");
	err = jtFile.write("tp_116 = suppressEmptyTextBox=" + suppressEmptyTextBox + "\n");
	err = jtFile.write("tp_117 = suppressEmptyImageBox=" + suppressEmptyImageBox + "\n");
	err = jtFile.write("tp_118 = suppressUnassignedBox=" + suppressUnassignedBox + "\n");
	err = jtFile.write("tp_119 = suppressLineBox=" + suppressLineBox + "\n");
	err = jtFile.write("tp_121 = suppressNonPrintingBox=" + suppressNonPrintingBox + "\n");

	err = jtFile.write("tp_122 = suppressExportMouseOvers=" + suppressExportMouseOvers + "\n");
	err = jtFile.write("tp_123 = CSSclassNamesASCII=" + CSSclassNamesASCII + "\n");
	err = jtFile.write("tp_124 = CSSfontFamilyString=" + CSSfontFamilyString + "\n");
	do {
		if (outputModeCustomXSL.toLowerCase().indexOf("basic") >= 0) { 
			err = jtFile.write("tp_125 = outputXMLComplexity=1\n");
			break;
		}
		if (outputModeCustomXSL.toLowerCase().indexOf("simple") >= 0) { 
			err = jtFile.write("tp_125 = outputXMLComplexity=2\n");
			break;
		}
		if (outputModeCustomXSL.toLowerCase().indexOf("medium") >= 0) { 
			err = jtFile.write("tp_125 = outputXMLComplexity=3\n");
			break;
		}
		err = jtFile.write("tp_125 = outputXMLComplexity=0\n");
	} while (false);
	err = jtFile.write("tp_126 = pageframingBoxNoExport=" + pageframingBoxNoExport + "\n");
	err = jtFile.write("tp_127 = pageframingBoxSizeThreshold=" + pageframingBoxSizeThreshold + "\n");
	err = jtFile.write("tp_128 = pageframingBoxesNoCatch=" + pageframingBoxesNoCatch + "\n");
	err = jtFile.write("tp_129 = fontsizeUnits=" + fontsizeUnits + "\n");
	err = jtFile.write("tp_130 = fontsizeBase=" + fontsizeBase + "\n");
	var metaViewportcont = metaViewportContent;
	/* DON'T recalculate! initial-scale=1.0 is fine
	if ((metaViewportInitialRescaleFactor > 0) && (parseInt(pageImageWidth) != 330)) {
		var newInitialScale = Math.ceil(metaViewportInitialRescaleFactor / parseFloat(pageImageWidth) * 100) / 100;	// round to 2 digits
		metaViewportcont = "initial-scale=" + newInitialScale + ", width=" + metaViewportWidth + ", user-scalable=" + metaViewportUserScalable;
	}
	*/
	err = jtFile.write("tp_131 = metaViewportContent=" + metaViewportcont + "\n");
	var inpPararam = imageInputParam;
	if (inpPararam == "") {
		if ((exportImages != 1) && (imageKeepTransparency != 0)) {	// do not set transparency for JPEG: bad loss of quality
			if (inpPararam != "") inpPararam += " ";
			inpPararam += "-background none";
			if (imageTransparentColor != "") {
				if (inpPararam != "") inpPararam += " ";
				inpPararam += "-transparent " + imageTransparentColor;
			}
		}
	}
	err = jtFile.write("tp_132 = imageInputParam=" + inpPararam + "\n");
	err = jtFile.write("tp_133 = imageDPI=" + imageInputDPI+","+imageDPI + "\n");
	err = jtFile.write("tp_134 = imageQUALITY=" + imageQUALITY + "\n");

	err = jtFile.write("tp_135 = websiteParams=" + websiteParams + "\n");
	err = jtFile.write("tp_136 = pageJPEGParams=" + pageJPEGParams + "\n");
	err = jtFile.write("tp_137 = pageJPEGfinish=" + pageJPEGfinish + "\n");

	err = jtFile.write("tp_138 = pub_companyName=" + companyName + "\n");
	err = jtFile.write("tp_139 = pub_objectName=" + objectName + "\n");
	if (isBatchMode) {	// we do not set this in batch mode because we could have exported many different documents
		err = jtFile.write("tp_140 = pub_objectShortcut=\n");
		err = jtFile.write("tp_141 = pub_issueYear=\n");
		err = jtFile.write("tp_142 = pub_issueDate=\n");
	}
	else {
		err = jtFile.write("tp_140 = pub_objectShortcut=" + objectShortcut + "\n");
		err = jtFile.write("tp_141 = pub_issueYear=" + issueYear + "\n");
		err = jtFile.write("tp_142 = pub_issueDate=" + issueDate + "\n");
	}

	err = jtFile.write("tp_143 = addVirtualKeyboard=" + addVirtualKeyboard + "\n");

	err = jtFile.write("tp_144 = imageAddTargetPrefix=" + imageAddTargetPrefix + "\n");
	err = jtFile.write("tp_145 = imageOverwriteTargetIfExists=" + imageOverwriteTargetIfExists + "\n");
	err = jtFile.write("tp_146 = imagesTotalSources=" + imgCnt + "\n");
	err = jtFile.write("tp_147 = imagesTotalSourcesMissing=" + imgMissing + "\n");
	err = jtFile.write("tp_148 = imageMissingCreateFromPagePreview=" + imageMissingCreateFromPagePreview + "\n");

	err = jtFile.write("tp_149 = extractImageMetaData=" + extractImageMetaData + "\n");
	err = jtFile.write("tp_150 = fullImageMetaData=" + fullImageMetaData + "\n");
	err = jtFile.write("tp_151 = typeImageMetaData=" + typeImageMetaData + "\n");
	err = jtFile.write("tp_152 = pageTurnMode=" + pageTurnMode + "\n");
	err = jtFile.write("tp_153 = excludeLayers=" + excludeLayers + "\n");
	err = jtFile.write("tp_154 = INDEXPHPNAME=index.php\n");
	err = jtFile.write("tp_155 = mainExportFolderIn=" + mainExportFolderInConst + "\n");
	err = jtFile.write("tp_156 = mainExportFolderOut=" + mainExportFolderOutConst + "\n");
	err = jtFile.write("tp_157 = PDFnoBoxCropping=12\n");	// Addable flags: 1 = don't set pdf:use-cropbox=true, 2 = don't set pdf:use-Trimbox=true

	err = jtFile.write("\n");
	err = jtFile.write("ftpUsername = \n");
	err = jtFile.write("ftpPassword = \n");
	err = jtFile.write("ftpPort = \n");
	err = jtFile.write("ftpEncoding = UTF-8\n");
	err = jtFile.write("\n");
	err = jtFile.write("newoutputFileNameExt = .xml\n");
	err = jtFile.write("excludeSourceProcessingRunFileNameExts = .xml,.incx,_int.xml,_indb.xml,.xmi\n");
	err = jtFile.write("sourceFileAction = " + pathAdapter(mainExportFolderDone) + folderSeparator + "\n");
	err = jtFile.write("deleteSourceDirs = 0\n");
	err = jtFile.write("nextJobTicketPath =\n");
	err = jtFile.write("nextJobTicketFileName = autostart.jt\n");
	err = jtFile.write("checkExtConverter = 0\n");

	err = jtFile.write("runBeforeJobApp = " + add_vars(runBeforeJobApp) + "\n");
	err = jtFile.write("runAfterJobApp = " + add_vars(runAfterJobApp) + "\n");
	err = jtFile.write("runBeforeTransformApp = " + add_vars(runBeforeTransformApp) + "\n");
	err = jtFile.write("runAfterTransformApp = " + add_vars(runAfterTransformApp) + "\n");

	err = jtFile.close();
	// for XML Tree, we do not want a second transform
	if ((outputModeCustomXSL == "") || (noHTMLconversion != "0") || (doingBookExport && (delete_individual_bookpagesHTM != 0))) {
		writeOverrideQueue(path,jtname);
		return;
	}




	// -----------------------------------------
	// write the second jobticket to transform the epaper xml file to html (like browser does)
	var jtname1 = new Date().getTime() + "override.jt1"
	jtFile = new File(commpath + jtname1);
	jtFile.encoding = "UTF-8";
	err = jtFile.open("w");

	err = jtFile.write("init_previous_jobticket = 0\n");
	err = jtFile.write("jobticketVersion = 10.0\n");
	err = jtFile.write("mode = 0\n");
	err = jtFile.write("\n");
	err = jtFile.write("jt_triggerfile = \n");
	if (callerScript.toLowerCase().indexOf("batch") >= 0) {	// Batch mode: state the path only
		err = jtFile.write("sourcePathName = " + pathAdapter(mainExportFolderOut) + folderSeparator + "\n");
	}
	else {
		if (theDocumentName != "") err = jtFile.write("sourcePathName = " + pathAdapter(mainExportFolderOut) + folderSeparator + theBaseDocumentName + ".xml\n");
		else err = jtFile.write("sourcePathName = " + pathAdapter(mainExportFolderOut) + folderSeparator + "\n");
	}
	err = jtFile.write("xslPathName = \n");
	err = jtFile.write("outputPathName = \n");
	var outExt = ".htm";
	do {
		var extExt = "", nameparts;
		nameparts = outputModeCustomXSL.split("_");
		if (nameparts.length > 1) {
			extExt = "_" + nameparts[nameparts.length-1];
			extExt = extExt.split(".")[0];
		}
		if (outputModeCustomXSL.toLowerCase().indexOf("2text") >= 0) { outExt = extExt + ".txt"; break; }
		if (outputModeCustomXSL.toLowerCase().indexOf("2xhtm") >= 0) { outExt = extExt + ".xhtml"; break; }
		if (outputModeCustomXSL.toLowerCase().indexOf("2xml") >= 0) { outExt = extExt + ".xml"; break; }
		if (outputModeCustomXSL.toLowerCase().indexOf("2htm") >= 0) { outExt = extExt + ".htm"; break; }
		if (outputModeCustomXSL.toLowerCase().indexOf("xml") == 0) { outExt = "_s.xml"; break; }
	} while(false);
	//alert(outputMode + "\n" +outputModeCustomXSL+ "\n\nnewoutputFileNameExt: " +outExt);

	err = jtFile.write("newoutputFileNameExt = " + outExt + "\n");
	err = jtFile.write("excludeSourceProcessingRunFileNameExts = .incx,_int.xml,_indb.xml,.xmi\n");
	err = jtFile.write("sourceFileAction = \n");
	err = jtFile.write("deleteSourceDirs = 0\n");
	err = jtFile.write("nextJobTicketPath =\n");
	err = jtFile.write("nextJobTicketFileName = autostart.jt\n");
	err = jtFile.write("checkExtConverter = 0\n");

	err = jtFile.close();

	writeOverrideQueue(path,jtname,jtname1);


	// -----------------------------------------
	// run hooks to write more JobTickets
	runHooks(scriptName,"afterWriteJobTickets", { "commpath":commpath });

	return;
}

function newMergeJobTicketBatchXSLT(compath, bookXMLPathName) {
	if (compath == null || compath == "") return;
	var f = new File(compath);
	//alert("newJobTicketBatchXSLT()\ncompath: " + compath +"\nf.fsName: " + f.fsName +"\nexists: " + f.exists);
	if (f.exists == false) {
		var fldr = new Folder(compath);
		fldr.create();
		f = new File(compath);
		if (f.exists == false) {
			alert(loc(25));
			return;
		}
	}

	// first, we merge all xml files from the book index file into one output XML file
	var commpath = compath;
	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	var jtname = new Date().getTime() + "override.jt2"
	jtFile = new File(commpath + jtname);
	jtFile.encoding = "UTF-8";
	err = jtFile.open("w");

	err = jtFile.write("init_previous_jobticket = 0\n");
	err = jtFile.write("jobticketVersion = 10.0\n");
	err = jtFile.write("mode = 0\n");
	err = jtFile.write("\n");
	err = jtFile.write("jt_triggerfile = \n");
	err = jtFile.write("sourcePathName = " + bookXMLPathName + "\n");
	err = jtFile.write("xslPathName = XSL/XMLpagesMerge.xsl\n");
	err = jtFile.write("outputPathName = \n");
	err = jtFile.write("newoutputFileNameExt = .xml\n");
//	err = jtFile.write("excludeSourceProcessingRunFileNameExts = _int.xml,_indb.xml,.xmi\n");
	err = jtFile.write("sourceFileAction = \n");
	err = jtFile.write("deleteSourceDirs = 0\n");
	err = jtFile.write("nextJobTicketPath =\n");
	err = jtFile.write("nextJobTicketFileName = autostart.jt\n");

	err = jtFile.write("tp_1 = delete_individual_bookpagesXML=" + delete_individual_bookpagesXML + "\n");
	err = jtFile.write("tp_2 = delete_individual_bookpagesHTM=" + delete_individual_bookpagesHTM + "\n");

	err = jtFile.close();

	// second, we convert the created full xml file to html
	var jtname1 = new Date().getTime() + "override.jt3"
	jtFile = new File(commpath + jtname1);
	jtFile.encoding = "UTF-8";
	err = jtFile.open("w");

	err = jtFile.write("init_previous_jobticket = 0\n");
	err = jtFile.write("jobticketVersion = 10.0\n");
	err = jtFile.write("mode = 0\n");
	err = jtFile.write("\n");
	err = jtFile.write("jt_triggerfile = \n");
	var fullBookXMLpath = bookXMLPathName;	// ends with .xmi : change to .xml
	fullBookXMLpath = fullBookXMLpath.substring(0, fullBookXMLpath.length - 1) + "l";
	err = jtFile.write("sourcePathName = " + fullBookXMLpath + "\n");
	err = jtFile.write("xslPathName = \n");
	err = jtFile.write("outputPathName = \n");
	err = jtFile.write("newoutputFileNameExt = .htm\n");
	err = jtFile.write("sourceFileAction = \n");
	err = jtFile.write("deleteSourceDirs = 0\n");
	err = jtFile.write("nextJobTicketPath =\n");
	err = jtFile.write("nextJobTicketFileName = autostart.jt\n");

	err = jtFile.close();

	writeOverrideQueue(commpath,jtname,jtname1);
}

function writeOverrideQueue(path,jtname,jtname1) {
	if ((path == null) || (path == "")) return(-1);
	var commpath = path;
	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	var queuename = "override.que";
	var queuepath = commpath + queuename;
	var xqueuename = "x" + queuename;
	var xqueuepath = commpath + xqueuename;
	jtQfile = new File(queuepath);
	var err = false;
	if (jtQfile.exists == true) {	// try to rename queue file if exists
		for (var i = 0; i < 1000; i++) {	// try to rename file
			err = jtQfile.rename(xqueuename);
			if (jtQfile.error != "") ;	//alert("rename fserror: " +jtQfile.error);
			if (err == true) break;
			//alert("locked");
			$.sleep(10);
		}
		if (err == false) {
			//alert("error rename to: " + xqueuename);
			return(-1);	// uuups
		}
	}

	jtQfile = new File(xqueuepath);
	jtQfile.encoding = "UTF-8";
	for (var i = 0; i < 100; i++) {	// try to open until file is free. don't wait too long
		err = jtQfile.open("e");
		if (err == true) break;
		$.sleep(100);
	}
	if (err == false) return(-1);
	if (jtQfile.error != "") alert("fserror: " +jtQfile.error);

	err = jtQfile.seek(0,2);	// seek to eof to append
	//if (err != true) alert("error seek");
	if ((jtname != null) && (jtname != "")) err = jtQfile.writeln(jtname);
	//if (err != true) alert("error jt write");
	if ((jtname1 != null) && (jtname1 != "")) err = jtQfile.writeln(jtname1);
	//if (err != true) alert("error jt1 write");
//	alert("wait to close file");
	err = jtQfile.close();
	//if (err != true) alert("error close");
	err = jtQfile.rename(queuename);
	//if (err != true) alert("error rename back to: " + queuename);
	if (err == false) return(-1);
	return(0);
}


/****************************
 * Change path from OSX to Windows style or vice versa
 */
function pathAdapter(path) {
	if ((path == null) || (path == "")) return("");
	var mypath = path;
	switch (os) {
		case "mac":
			if (BXSLT4InDesignApplRunsOnWindows == "1") {
				if (mypath.indexOf(servernameOnOSX) == 0) mypath = mypath.replace(servernameOnOSX,servernameOnWindows);
				mypath = mypath.replace(/\//gi,"\\");
				break;
			}
			if (BXSLT4InDesignApplRunsOnUnix == "1") {
				if (mypath.indexOf(servernameOnOSX) == 0) mypath = mypath.replace(servernameOnOSX,servernameOnUnix);
				mypath = mypath.replace(/\//gi,"\\");
				break;
			}
			break;
		case "win":
			if (BXSLT4InDesignApplRunsOnOSX == "1") {
				if (mypath.indexOf(servernameOnWindows) == 0) mypath = mypath.replace(servernameOnWindows, servernameOnOSX);
				mypath = mypath.replace(/\\/gi,"/");
				break;
			}
			if (BXSLT4InDesignApplRunsOnUnix == "1") {
				if (mypath.indexOf(servernameOnWindows) == 0) mypath = mypath.replace(servernameOnWindows, servernameOnUnix);
				mypath = mypath.replace(/\\/gi,"/");
				break;
			}
			break;
	}
	return(mypath);
}


/****************************
 * check if tranformer application may be found
 */
function findBXSLTappl() {
	var found = false,
		runcnt = 0;
	do {
		runcnt++;
		switch (runcnt) {
			case 1:	// try the path currently set by loadCommPrefsFromFile
				// BXSLT4InDesignBasePath already IS set
				break;
			case 2:	// try default path in user.home - may be a downloaded version for testing only
				BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault;
				break;
			case 3:	// try default path in /Applications (OSX) or %Programfiles% (WIN)
				BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault2;
				break;
			case 4:	// try default path in %Programfiles% (WIN) without ' (x86)'
				if (os == 'win') {
					var bp = BXSLT4InDesignBasePathDefault2
					if (bp.indexOf(' (x86)') >= 0) bp = bp.replace(' (x86)','');
					BXSLT4InDesignBasePath = bp;
				} else continue;
				break;
			case 5:	// try to find any folder in /Applications containing 'applicationProgramName*'
				var fldr = new Folder(programFolder);
				var applFldrs = fldr.getFiles(applicationProgramName + packageExtension + "*");
				if (applFldrs.length < 1) break;	// let user choose the application
				for (var i = 0; i < applFldrs.length; i++) {
					var afldr = applFldrs[i];
					var applfldr = afldr + folderSeparator + tranformerName + folderSeparator + tranformerName + ".app";
					//alert(applfldr);
					fldr = new File(applfldr);
					if (fldr.exists == true) {
						BXSLT4InDesignBasePath = afldr + folderSeparator + tranformerName + folderSeparator;
						//alert("found appl: " + applfldr);
						break;
					}
				}
				break;	// let user choose the application
			default:	// this will open the file chooser
		}
		// set transformer application to call
		BXSLT4InDesignPath = BXSLT4InDesignBasePath + BXSLT4InDesignAppl;
		//alert("runcnt:"+runcnt +"\n"+ BXSLT4InDesignPath);
		var f = new File(BXSLT4InDesignPath);
		if (f.exists == false) {
			if (runcnt < 5) continue;
			//alert("The '" + applicationHumanName + "' application can not be found at\n" + BXSLT4InDesignPath + "\n\nPlease, copy the application to your HOME folder (as mentioned in the manual) or choose the application in the following dialog and try again!");
			alert(loc(2));
			var bxslt_appl_fldr = Folder.selectDialog(loc(3));	// "Please select the folder containing the tranformer application"
			if (bxslt_appl_fldr == null) {
				//alert("Document export aborted.\n\nThe application may not continue exporting your document because the Transformer application may not be found.");
				alert(loc(4));
				return(false);
			}
			if (bxslt_appl_fldr != null) BXSLT4InDesignBasePath = bxslt_appl_fldr + folderSeparator;
			continue;
		}
		found = true;
		break;
	} while(true);
	return(found);
}


/****************************
 * check if exists or create main export folder
 */
function create_folderstructure(thepath) {
	if (thepath == "") return(error.NOERR);	// nothing to check = noerror
	var f = new Folder(thepath);
	f.create();
	//alert("create_folderstructure()\nthepath: " + thepath + "\nf.exists: " + f.exists);
	last_errorCode = error.NOERR;
	if (f.exists == false) {
		last_errorCode = error.EXPORT_FOLDERSTRUCT_NOT_FOUND;
		var folder_names = thepath.split(folderSeparator);
		var fldr = null;
		var fldr_path = "";
		for (var i = 0; i < folder_names.length; i++) {
			if (folder_names[i] == "") continue;
			fldr_path += folder_names[i] + folderSeparator;
			fldr = new Folder(fldr_path);
			fldr.create();
		}
		f = new File(thepath);	// recheck for existance
		if (f.exists == false) {
			//alert("The export folder structure could not be created:\n" + mainExportFolderIn + "\n\nPlease, create it manually.");
			if ( (silentMode >= 0) && !(silentMode & 2) ) alert(loc(7));
			last_errorCode = error.EXPORT_FOLDERSTRUCT_NOT_CREATED;
			return(error.EXPORT_FOLDERSTRUCT_NOT_CREATED);
		}
		last_errorCode = error.NOERR;
	}
	return(error.NOERR);
}


/****************************
 * copy a document's images to the export path
 */
function copyImages(toPath) {
	imgCnt = 0;
	imgMissing = 0;
	imgOK = 0;
	var img = null;
	var numimages = app.activeDocument.links.length;
	if (numimages <= 0) return;
	if (!(silentMode & 4)) {
		messageClose();
		message(loc(19,numimages));	// Copying images ...
	}
	for(imgCnt = 0; imgCnt < numimages; imgCnt++){
		img = app.activeDocument.links.item(imgCnt);
		//alert("image #" + imgCnt + ":\n" + img.filePath);
		f = new File(img.filePath);
		if (f.exists == true) {
			var targfile = toPath + folderSeparator;
			var imgname = img.name;
			if (imgname.indexOf("/") >= 0) imgname = imgname.replace(/\//g,":");
			if (totalencode == false) targfile += xPlatformName(imgname,false);
			else targfile += xPlatformName(imgname,true);
			var ftarg = new File(targfile);
			var success = false;
			if (ftarg.exists == false) {	// do not copy if already there (same image placed multiple times)
				success = f.copy(targfile);
				//alert("image copy #" + imgCnt + " success '" + success + ":'\nsource: '" + img.filePath + "'\ntarget: '" + targfile + "'");
			}
			else {
				//alert("image copied #" + imgCnt + ":\n" + img.filePath);
				success = true;
			}
			if (success) imgOK++;
			else {
				//alert("image not copied #" + imgCnt + ":\n" + img.filePath);
				imgMissing++;
			}
		}
		else {
			//alert("image missing #" + imgCnt + ":\n" + img.filePath);
			if (exportImages > 0) {  // 0 = no image export
			                         // 1 = JPEG
			                         // 2 = PNG
			  var created = false;
			  if (imageMissingCreateFromPagePreview == 1) created = create_image_from_box(img, toPath);
			  if (created) imgOK++;
			  else imgMissing++;
			}
		}
	}
	messageClose();
}

function is_imageExclude(name, excludeList) {
	if ((excludeList == null) || (excludeList == "")) {
	  return(false);
	}
	if ((name == null) || (name == "")) {
	  return(false);
	}
  var imageExclude = excludeList.split(";");
	if (imageExclude.length == 0) return(false);

	for (var i = 0; i < imageExclude.length; i++) {
		if (name.indexOf(imageExclude[i]) == 0) return(true);
	}
	return(false);
}

function create_all_images_from_box(toPath) {
	imgCnt = 0;
	imgMissing = 0;
	imgOK = 0;
	var img = null;
	var numimages = app.activeDocument.links.length;
	if (numimages <= 0) return;
	if (!(silentMode & 4)) {
		messageClose();
		message(loc(26,numimages));	// Reproducing images ...
	}
	
	// the image sizes
	var imgparmsArr = imageParameters.split("//");

  // export all images
  var imgcontainer = null;
  var imgnameOrig = "";
  var imgname1 = "";
  var imgname2 = "";
	for(imgCnt = 0; imgCnt < numimages; imgCnt++){
		img = app.activeDocument.links.item(imgCnt);

    if (is_imageExclude(img.name, imageEXCLUDE)) continue;

		imgcontainer = img.parent.parent;
		imgnameOrig = img.name;
    imgname1 = "";
    imgname2 = "";

		// create first image
		imgname1 = create_image_from_box(img, toPath, null);
		//alert("imgname1: "+imgname1);
		// evtl. create more images at different size
    if (imgparmsArr.length > 1) {
		  imgname2 = create_image_from_box(img, toPath, imgparmsArr[1]);
    }
    // check for converted images
		if (imgname1 != "") {
		  var newlabel = "";
		  var label = "##imgnameorig:" + imgnameOrig.replace(/"/,"\"") + "##"
      if (imgcontainer.label.indexOf(label) < 0) newlabel += label;

		  label = "imgname1:" + imgname1 + "##";
      if (imgcontainer.label.indexOf(label) < 0) newlabel += label;

		  if (imgname2 != "") {
		    label = "imgname2:" + imgname2 + "##";
        if (imgcontainer.label.indexOf(label) < 0) newlabel += label;
		  }

		  if (newlabel != "") imgcontainer.label += newlabel;
		  
		  imgOK++;
		}
		else imgMissing++;
	}
	messageClose();
}



function scanPagesForAnchoredGraphics() {

	function expandLabel(origLabel, addLabel) {
		var newLabel = "" + addLabel;
		if (startsWith(newLabel,"##") == false) newLabel = "##"+newLabel;
		if (endsWith(newLabel,"##") == false) newLabel = newLabel+"##";
		
		if (origLabel.indexOf(newLabel) >= 0) return(origLabel);
	
		return (origLabel + newLabel);
	}

	var theDocument = null,
		totalPages = 0,
		totalTextFrames = 0,
		totalStoryGraphics = 0;
	var DEBUG = 0;
	
	var measurementunit_orig = app.scriptPreferences.measurementUnit;
	app.scriptPreferences.measurementUnit = MeasurementUnits.MILLIMETERS;
	var measureFactor = 72/25.4;
	var measurementunit = "px";


	app.selection = NothingEnum.nothing;
	theDocument = app.activeDocument;
	theDocument.zeroPoint = new Array(0.0,0.0);	// make sure zero point is at top left page corner

	try { theDocument.layers.everyItem().locked = false; } catch(e){}
	
	totalPages = theDocument.pages.length;

	var pagectr = 0;
	for (pagectr = 0; pagectr < totalPages; pagectr++) {
		var curpage = theDocument.pages.item(pagectr);

		var numpageTextFrames = theDocument.pages.item(pagectr).textFrames.length;
		if (numpageTextFrames <= 0) continue;

		for (var framectr = 0; framectr < numpageTextFrames; framectr++) {
			totalTextFrames++;
			var theTextFrame = theDocument.pages.item(pagectr).textFrames.item(framectr);
			// get first boxes of a chain only
			var theStory = theTextFrame.parentStory;
			var allStoryGraphics = theStory.allGraphics;	// Array of Graphic
			if (allStoryGraphics.length <= 0) continue;
			totalStoryGraphics += allStoryGraphics.length;
			var allStoryPageItems = theStory.allPageItems;	// Array of PageItem
			//alert("allStoryGraphics.length: " + allStoryGraphics.length +"\ntotalStoryGraphics: "+totalStoryGraphics);

			for (var graphicsctr = 0; graphicsctr < allStoryGraphics.length; graphicsctr++) {
				var storygraphic = allStoryGraphics[graphicsctr];	// returns the graphic frame
				var curlabel = storygraphic.label;
				var bounds = null;
				try {
					bounds = (storygraphic.parent.geometricBounds[1]*measureFactor)+","+(storygraphic.parent.geometricBounds[0]*measureFactor)+","+(storygraphic.parent.geometricBounds[3]*measureFactor)+","+(storygraphic.parent.geometricBounds[2]*measureFactor);
				} catch (e) {
					continue;
				}
				var addlabel = "ag_gb:"+bounds+","+measurementunit+","+storygraphic.parentPage.name;
				var newlabel = expandLabel(curlabel, addlabel);
				storygraphic.label = newlabel;
				if (DEBUG > 0) alert("allStoryGraphics.length: " + allStoryGraphics.length + "\ngraphicsctr: " + graphicsctr + "\nid: " + storygraphic.id + "\ngeometricBounds: " + storygraphic.geometricBounds + "\nnewlabel: " + newlabel);
			}
		}

	}

	if (DEBUG > 0) {
		alert("*** anchored graphics:"
			+ "\ntotalPages: " + totalPages
			+ "\ntotalTextFrames: " + totalTextFrames
			+ "\ntotalStoryGraphics (anchored): " + totalStoryGraphics
			);
	}

	app.scriptPreferences.measurementUnit = measurementunit_orig;

}

function create_image_from_box(link, toPath, size) {

  var myXUnits = null, myYUnits = null;
  var imgWidth = null, imgHeight = null;
	var container = link.parent.parent;
	var boxid = "u"+d2hs(container.id)+"_";
	var target_name = xPlatformName(link.name,false);
	target_name = get_image_name(target_name, null); // cut name extension
	if (size != null) target_name += "_" + size;
	
	var currentDocument = app.activeDocument;
  try {
		myXUnits = currentDocument.viewPreferences.horizontalMeasurementUnits;
		myYUnits = currentDocument.viewPreferences.verticalMeasurementUnits;
	
		imgWidth = container.geometricBounds[3] - container.geometricBounds[1];
		imgHeight = container.geometricBounds[2] - container.geometricBounds[0];
		imgWidth = new UnitValue (imgWidth, myXUnits).as('px');
		imgHeight = new UnitValue (imgHeight, myYUnits).as('px');
		if (size != null) {
			imgWidth = imgWidth * parseFloat(size);
			imgHeight = imgHeight * parseFloat(size);
		}
		imgWidth = Math.floor(imgWidth);
		imgHeight = Math.floor(imgHeight);
	} catch(e){}


	// geometricBounds: The bounds of the Rectangle excluding the stroke width, in the format [y1, x1, y2, x2], which give the coordinates of the top-left and bottom-right corners of the bounding box.
	/*
	alert(target_name+"\n"
	      +container.geometricBounds + "\n"
	      + "UnitsX: " + myXUnits + ", UnitsY: " + myYUnits + "\n"
	      + "imgWidth px: " + imgWidth + ", imgHeight px: " + imgHeight + "\n");
	*/
	target_name = boxid + target_name;

	switch (exportImages) {
	  case 1: // JPEG
	    target_name += ".jpg";
	    break;
	  case 2: // PNG
	    target_name += ".png";
	    break;
	}  

	var myoutpath = toPath;
	if (endsWith(myoutpath,folderSeparator) == false) myoutpath += folderSeparator;
	var outpath = myoutpath + target_name;
	
	// calculate image DPI
	var theImageDPI = parseInt(imageDPI);
	if (size != null) {
	  theImageDPI = parseInt(imageDPI * parseFloat(size));
	}
	/*
	alert("link.name: " + link.name
			+ "\nbounds graphic: "+link.parent.geometricBounds
			+ "\nbounds box: "+link.parent.parent.geometricBounds
			+ "\ntarget_name: "+target_name
			+ "\noutpath: "+outpath
			);
	*/
	switch (exportImages) {
	  case 1: // JPEG
			// export image to JPG
			try { app.jpegExportPreferences.antiAlias = true; } catch (e) {}
			try { app.jpegExportPreferences.embedColorProfile = imageJPEGembedColorProfile; } catch (e) {}
			try { app.jpegExportPreferences.exportResolution = theImageDPI; } catch (e) {}
			try { app.jpegExportPreferences.jpegColorSpace = JpegColorSpaceEnum.RGB; } catch (e) {}
			try { app.jpegExportPreferences.jpegQuality = JPEGOptionsQuality.HIGH; } catch (e) {}  // LOW, MEDIUM, HIGH, MAXIMUM
			try { app.jpegExportPreferences.jpegRenderingStyle = JPEGOptionsFormat.baselineEncoding; } catch (e) {} 	// BASELINEENCODING or PROGRESSIVE_ENCODING
	    break;
	  case 2: // PNG
			// export image to PNG
			try { app.pngExportPreferences.antiAlias = true; } catch (e) {}
			//try { app.pngExportPreferences.embedColorProfile = imageJPEGembedColorProfile; } catch (e) {}
			try { app.pngExportPreferences.exportResolution = theImageDPI; } catch (e) {}
			try { app.pngExportPreferences.pngColorSpace = PNGColorSpaceEnum.RGB; } catch (e) {}
			try { app.pngExportPreferences.pngQuality = PNGQualityEnum.HIGH; } catch (e) {}  // LOW, MEDIUM, HIGH, MAXIMUM
			try { app.pngExportPreferences.transparentBackground = true; } catch (e) {}  // with transparency
			try { app.pngExportPreferences.simulateOverprint = false; } catch (e) {}  // with overprint simulation
	    break;  
	}

	var ftarg = new File(outpath);

	switch (exportImages) {
	  case 1: // JPEG
    	container.exportFile (ExportFormat.JPG , ftarg, false);
	    break;
	  case 2: // PNG
    	container.exportFile (ExportFormat.PNG_FORMAT , ftarg, false);
	    break;  
	}

	if (ftarg.exists == true) {
    // OK: return image name / imgWidth / imgHeight 
    if (imgWidth != null) target_name += "/" + imgWidth;
    if (imgHeight != null) target_name += "/" + imgHeight;
	  return(target_name);
	}
	//alert("link.name: " + link.name + "\nnot reproduced");
	return("");
}

function d2hs(d) {	// decimal to hex with small letters
	var h = hDs.substr(d&15,1);
	while (d>15) { d>>=4; h=hDs.substr(d&15,1)+h;}
	return h;
}
function get_image_name(filename, new_ext) {
	if ( (filename == null) || (filename == "") ) {
		return "";
	}
	var image_filename = filename.replace(/~sep~/g,"_");

	if ( (new_ext == null) || (new_ext != "") ) {
		var name = image_filename;
		var lastDot = name.lastIndexOf(".");
		if (lastDot >= 0) {
			namepart = name.substring(0,lastDot);
			extpart = name.substr(lastDot+1,name.length);
			if (is_valid_image_extension(extpart)) {
				name = namepart;
			}
		}
		if (new_ext == null) { // we just cut off the extension
		  return(name);
		}
		image_filename = name + new_ext;
	}
	return(image_filename);
}
function is_valid_image_extension(ext) {
	if (ext.length < 2) return(false);
	if (ext.length > 5) return(false);
	var myext = ext.toLowerCase();
	if (myext.indexOf('tif') == 0) return (true);
	if (myext.indexOf('eps') == 0) return (true);
	if (myext.indexOf('psd') == 0) return (true);
	if (myext.indexOf('pdf') == 0) return (true);
	if (myext.indexOf('ai') == 0) return (true);
	if (myext.indexOf('jpg') == 0) return (true);
	if (myext.indexOf('png') == 0) return (true);
	if (myext.indexOf('gif') == 0) return (true);
	if (myext.indexOf('bmp') == 0) return (true);
	if (myext.indexOf('svg') == 0) return (true);
	return(false);
}




/****************************
 * how the Alert to save the document or to continue anyway
 */
function showSaveDialog() {
	var myDialog = app.dialogs.add({name:windowTitle, canCancel:true});
	with(myDialog){
		with(dialogColumns.add()){
			with(dialogRows.add()){
				staticTexts.add({staticLabel:loc(6)});
			}
			with(dialogRows.add()){
				staticTexts.add({staticLabel:loc(23)});
			}
			with(dialogRows.add()){
				staticTexts.add({staticLabel:loc(24)});
			}
		}
	}
	//Display the dialog box.
	var go = myDialog.show();
	if (go == false) return(false); // cancelled dialog means to continue anyway
	return(true);
}






/****************************
 * Show the main export dialog for CS4 and newer
 */
function showMainDialog() {
	var mainExpfldr = new Folder(mainExportFolder);
	var infldrConst = new Folder(mainExportFolderInConst);
	var infldr = new Folder(mainExportFolderInConst);
	var outfldr = new Folder(mainExportFolderOutConst);
	var trans = new Folder(BXSLT4InDesignBasePath);
	var commfldr = new Folder(BXSLT4InDesignCommDir);
	var settingsfldr = new Folder(settingsFilePath);
	// issue date formats
	var issueDateFormatSelected = 0;
	if ((fileDateFormat != "") && (fileDateFormat != "YYYYMMDD")) {
		for (var si = 0; si < issueDateFormatStringList.length; si++) {
			if (issueDateFormatStringList[si] == fileDateFormat) {
				issueDateFormatSelected = si;
				break;
			}
		}
	}
	//alert(availableOutputXSLs);
	// create the popup string list 
	var outputModeStringList = [loc(271),loc(273),loc(274),loc(275)];	//removed 272: 'pages epaper and 276 slidebook
	// ad the custom ones
	for (var i = 0; i < availableOutputXSLs.length; i++) {
		if (	(availableOutputXSLs[i] == "flipbook.xsl")
			 || (availableOutputXSLs[i] == "xml2htmlEpaper.xsl")
			 || (availableOutputXSLs[i] == "xml2htmlList.xsl")
			 || (availableOutputXSLs[i] == "xml2htmlSimpleList.xsl")
			 || (availableOutputXSLs[i] == "")
			 || (availableOutputXSLs[i] == "slidebook.xsl")
			) {
			continue;
		}
		outputModeStringList[outputModeStringList.length] = availableOutputXSLs[i];
	}
	var outputModeSelected = outputMode;
	if (outputModeCustomXSL != "") {
		for (var si = 0; si < outputModeStringList.length-1; si++) {
			if (outputModeStringList[si] == outputModeCustomXSL) {
				outputModeSelected = si;
				break;
			}
		}
	}
	if (outputModeSelected >= outputModeStringList.length) {
		outputMode = outputModeSelected = 0;
		outputModeCustomXSL == "";
	}

	windowTitle = applicationHumanName + " - XML Export  v" + version + "    '" +  settingsFileName + "'    " + copyright;

	var dlgw = new Window('dialog', windowTitle, undefined, {resizeable:false, closeButton: true, maximizeButton:false, minimizeButton:false});
	dlgw.orientation = 'row';
	dlgw.margins = [7,7,7,7];
	dlgw.alignChildren = ['left','top'];
	dlgw.alignment = ['left','top'];
	
	var windowFont = dlgw.graphics.font;
	var font = ScriptUI.newFont ( windowFont.name, windowFont.name.style, 11 );
	dlgw.graphics.font = font;
	//alert("Font name: " + font.name + "\n    style: " + font.style + "\n    size: " + font.size);
	var redpen = dlgw.graphics.newPen (dlgw.graphics.PenType.SOLID_COLOR, [1,0,0,1], 1);
	var labelSize = [115,2.0*font.size];
	var labelSize2 = [200,4.0*font.size];
	var editSize = [105,2.0*font.size];
	var editSize2 = [150,2.0*font.size];
	var editSize5chars = [47,2.0*font.size];
	var editSize15chars = [150,2.0*font.size];
	var editSize3chars = [35,2.0*font.size];

	var pathInfosLeftColumnWidth = 170;
	var pathInfosRightColumnWidth = 380;
	var pathInfosSize = [pathInfosLeftColumnWidth,Math.ceil(2.0*font.size)];
	var pathInfosSize2 = [pathInfosRightColumnWidth,2.0*font.size];
	var pathEditSize2 = [pathInfosRightColumnWidth,2.0*font.size];
	var pathInfosCheckSize = [pathInfosLeftColumnWidth,Math.ceil(2.0*font.size)];
	var pathInfosButtonSize = [pathInfosLeftColumnWidth,Math.ceil(2.0*font.size)];
	var dotdotWidth = 55;
	var dropDownMaxWidth1 = 180;
	var dropDownMaxWidth2 = 350;
	var indent = 18;

		//------------------------------
		// items on left side
		dlgw.leftGroup = dlgw.add('group');
		dlgw.leftGroup.orientation = 'column';
		dlgw.leftGroup.spacing = 5;
/*
			if (settingsloaded != 0) {
				dlgw.leftGroup.errMsgGroup2 = dlgw.leftGroup.add('panel', undefined);
				dlgw.leftGroup.errMsgGroup2.alignChildren = ['left','top'];
				dlgw.leftGroup.errMsgGroup2.alignment = ['center','top'];
				dlgw.leftGroup.errMsgGroup2.spacing = 0;
				dlgw.leftGroup.errMsgGroup2.margins = [15,3,15,3];
				dlgw.leftGroup.errMsgGroup2.message = dlgw.leftGroup.errMsgGroup2.add('statictext', undefined, loc(84), {multiline:true});	// Settings File not reachable
					dlgw.leftGroup.errMsgGroup2.message.graphics.font = font;
					dlgw.leftGroup.errMsgGroup2.message.graphics.foregroundColor = redpen;
			}
*/
			//------------------------------
			// doc info and paths
			dlgw.leftGroup.basePathGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.basePathGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.basePathGroup.alignment = ['left','top'];
				dlgw.leftGroup.basePathGroup.label = dlgw.leftGroup.basePathGroup.add('statictext', undefined, loc(30));
					dlgw.leftGroup.basePathGroup.label.graphics.font = font;
				dlgw.leftGroup.basePathGroup.basePath = dlgw.leftGroup.basePathGroup.add('statictext', undefined, infldrConst.fsName);
					dlgw.leftGroup.basePathGroup.basePath.graphics.font = font;
			dlgw.leftGroup.companyNameGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.companyNameGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.companyNameGroup.alignment = ['left','top'];
				dlgw.leftGroup.companyNameGroup.label = dlgw.leftGroup.companyNameGroup.add('statictext', undefined, loc(31));
					dlgw.leftGroup.companyNameGroup.label.graphics.font = font;
					dlgw.leftGroup.companyNameGroup.label.size = labelSize;
				dlgw.leftGroup.companyNameGroup.companyNameField = dlgw.leftGroup.companyNameGroup.add('edittext', undefined, companyName);
					dlgw.leftGroup.companyNameGroup.companyNameField.graphics.font = font;
					dlgw.leftGroup.companyNameGroup.companyNameField.size = editSize;
					dlgw.leftGroup.companyNameGroup.companyNameField.helpTip = loc(150);	// Company name owning this object
				dlgw.leftGroup.companyNameGroup.info = dlgw.leftGroup.companyNameGroup.add('statictext', undefined, loc(32));
					dlgw.leftGroup.companyNameGroup.info.graphics.font = font;
			dlgw.leftGroup.objectNameGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.objectNameGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.objectNameGroup.alignment = ['left','middle'];
			dlgw.leftGroup.objectNameGroup.orientation = "row";
				dlgw.leftGroup.objectNameGroup.label = dlgw.leftGroup.objectNameGroup.add('statictext', undefined, loc(33));
					dlgw.leftGroup.objectNameGroup.label.size = labelSize;
					dlgw.leftGroup.objectNameGroup.label.graphics.font = font;
				dlgw.leftGroup.objectNameGroup.objectNameField = dlgw.leftGroup.objectNameGroup.add('edittext', undefined, objectName);
					dlgw.leftGroup.objectNameGroup.objectNameField.size = editSize;
					dlgw.leftGroup.objectNameGroup.objectNameField.graphics.font = font;
					dlgw.leftGroup.objectNameGroup.objectNameField.helpTip = loc(151);	// Name of this object
				var spacer1 = dlgw.leftGroup.objectNameGroup.add('statictext', undefined, ""); spacer1.size = [15,10];	// spacer
				dlgw.leftGroup.objectNameGroup.info = dlgw.leftGroup.objectNameGroup.add('statictext', undefined, loc(34));
					dlgw.leftGroup.objectNameGroup.info.graphics.font = font;
			dlgw.leftGroup.objectShortcutGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.objectShortcutGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.objectShortcutGroup.alignment = ['left','top'];
				dlgw.leftGroup.objectShortcutGroup.label = dlgw.leftGroup.objectShortcutGroup.add('statictext', undefined, loc(35));
					dlgw.leftGroup.objectShortcutGroup.label.size = labelSize;
					dlgw.leftGroup.objectShortcutGroup.label.graphics.font = font;
				dlgw.leftGroup.objectShortcutGroup.objectShortcutField = dlgw.leftGroup.objectShortcutGroup.add('edittext', undefined, objectShortcut);
					dlgw.leftGroup.objectShortcutGroup.objectShortcutField.size = editSize;
					dlgw.leftGroup.objectShortcutGroup.objectShortcutField.graphics.font = font;
					dlgw.leftGroup.objectShortcutGroup.objectShortcutField.helpTip = loc(152);	// Object's shortcut
				var spacer2 = dlgw.leftGroup.objectShortcutGroup.add('statictext', undefined, ""); spacer2.size = [15,10];	// spacer
				dlgw.leftGroup.objectShortcutGroup.info = dlgw.leftGroup.objectShortcutGroup.add('statictext', undefined, loc(36));
					dlgw.leftGroup.objectShortcutGroup.info.graphics.font = font;
			dlgw.leftGroup.issueYearGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.issueYearGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.issueYearGroup.alignment = ['left','top'];
				dlgw.leftGroup.issueYearGroup.label = dlgw.leftGroup.issueYearGroup.add('statictext', undefined, loc(37));
					dlgw.leftGroup.issueYearGroup.label.size = labelSize;
					dlgw.leftGroup.issueYearGroup.label.graphics.font = font;
				dlgw.leftGroup.issueYearGroup.issueYearField = dlgw.leftGroup.issueYearGroup.add('edittext', undefined, issueYear);
					dlgw.leftGroup.issueYearGroup.issueYearField.size = editSize;
					dlgw.leftGroup.issueYearGroup.issueYearField.graphics.font = font;
					dlgw.leftGroup.issueYearGroup.issueYearField.helpTip = loc(153);	// 4 digits year
				var spacer2 = dlgw.leftGroup.issueYearGroup.add('statictext', undefined, ""); spacer2.size = [15,10];	// spacer
				dlgw.leftGroup.issueYearGroup.info = dlgw.leftGroup.issueYearGroup.add('statictext', undefined, loc(38));
					dlgw.leftGroup.issueYearGroup.info.graphics.font = font;
			dlgw.leftGroup.issueDateGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.issueDateGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.issueDateGroup.alignment = ['left','top'];
				dlgw.leftGroup.issueDateGroup.label = dlgw.leftGroup.issueDateGroup.add('statictext', undefined, loc(39));
					dlgw.leftGroup.issueDateGroup.label.size = labelSize;
					dlgw.leftGroup.issueDateGroup.label.graphics.font = font;
				dlgw.leftGroup.issueDateGroup.issueDateField = dlgw.leftGroup.issueDateGroup.add('edittext', undefined, issueDate);
					dlgw.leftGroup.issueDateGroup.issueDateField.size = editSize;
					dlgw.leftGroup.issueDateGroup.issueDateField.graphics.font = font;
					dlgw.leftGroup.issueDateGroup.issueDateField.helpTip = loc(154);	// 8 digits full date
				var spacer2 = dlgw.leftGroup.issueDateGroup.add('statictext', undefined, ""); spacer2.size = [15,10];	// spacer
				dlgw.leftGroup.issueDateGroup.info = dlgw.leftGroup.issueDateGroup.add('statictext', undefined, loc(40));
					dlgw.leftGroup.issueDateGroup.info.graphics.font = font;
				dlgw.leftGroup.issueDateGroup.issueDateFormatDropdown = dlgw.leftGroup.issueDateGroup.add('dropdownlist', undefined, '', {items:issueDateFormatStringList});	// issue dates formats
					dlgw.leftGroup.issueDateGroup.issueDateFormatDropdown.items[issueDateFormatSelected].selected = true;
					dlgw.leftGroup.issueDateGroup.issueDateFormatDropdown.graphics.font = font;
					dlgw.leftGroup.issueDateGroup.issueDateFormatDropdown.helpTip = loc(301);	// XSL to add for 

			if (mainDialogErrorMessage != "") {	// show the error message
				dlgw.leftGroup.errMsgGroup = dlgw.leftGroup.add('panel', undefined);
				dlgw.leftGroup.errMsgGroup.alignChildren = ['left','top'];
				dlgw.leftGroup.errMsgGroup.alignment = ['center','top'];
				dlgw.leftGroup.errMsgGroup.margins = [15,3,15,3];
					var message = loc(22,mainDialogErrorCode,mainDialogErrorMessage);
					dlgw.leftGroup.errMsgGroup.errormessage = dlgw.leftGroup.errMsgGroup.add('statictext', undefined, message);
					dlgw.leftGroup.errMsgGroup.errormessage.graphics.font = font;
					dlgw.leftGroup.errMsgGroup.errormessage.graphics.foregroundColor = redpen;
			}


			//------------------------------
			// add options tabbed panel
			dlgw.leftGroup.tabPanel = dlgw.leftGroup.add('tabbedpanel', undefined, '');
			dlgw.leftGroup.tabPanel.alignChildren = ['fill','fill'];
			dlgw.leftGroup.tabPanel.alignment = ['fill','fill'];
			dlgw.leftGroup.tabPanel.size = [630,300];

				//------------------------------
				// output mode and pages stuff
				dlgw.leftGroup.tabPanel.tab1 = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(155));	// Output Types
				dlgw.leftGroup.tabPanel.tab1.orientation = 'row';
					dlgw.leftGroup.tabPanel.tab1.sub1Group = dlgw.leftGroup.tabPanel.tab1.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab1.sub1Group.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab1.sub1Group.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.tab1.sub1Group.orientation = 'column';
						dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.alignChildren = ['left','middle'];
							dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeLabel = dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.add('statictext', undefined, loc(270));	// output mode:
								dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeLabel.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown = dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.add('dropdownlist', undefined, '', {items:outputModeStringList});	// output mode:
								dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown.items[outputModeSelected].selected = true;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown.helpTip = loc(166);	// XSL to add for 
							// initial page turn mode: 'turn' or 'slide'
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.indent = 95;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.turnRadio = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.add('radiobutton', undefined, loc(280));
									dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.turnRadio.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.turnRadio.value = (pageTurnMode=="turn");
									dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.turnRadio.helpTip = loc(281);	// "Initial mode: turning pages"
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.slideRadio = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.add('radiobutton', undefined, loc(282));
									dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.slideRadio.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.slideRadio.value = (pageTurnMode=="slide");
									dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.slideRadio.helpTip = loc(283);	// "Initial mode: sliding pages"
						
						dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.alignChildren = ['left','middle'];
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.add("checkbox", undefined, loc(47));
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.value = exportPageJPEGs;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.helpTip = loc(167);	// Page Preview JPEG

						dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1 = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.indent = 20;

							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.atWidthLabel = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('statictext', undefined, loc(48));	// at width
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.atWidthLabel.graphics.font = font;
							var mypageImageWidth = pageImageWidth.replace(/x/gi,"");	// remove trailing 'x'
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageWidthField = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('edittext', undefined, mypageImageWidth);	// at width value
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageWidthField.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageWidthField.size = editSize15chars;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageWidthField.helpTip = loc(168);	// Width of Page Preview JPEG
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pxLabel = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('statictext', undefined, loc(49));	// px
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pxLabel.graphics.font = font;

							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageDPIField = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('edittext', undefined, pageImageDPI);	// DPI value
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageDPIField.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageDPIField.size = editSize3chars;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageDPIField.helpTip = loc(320);	// Page preview JPEG DPI
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.dpiLabel = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('statictext', undefined, loc(321));	// dpi
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.dpiLabel.graphics.font = font;

							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageQualityField = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('edittext', undefined, pageImageQuality);	// JPEG quality
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageQualityField.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageQualityField.size = editSize3chars;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageQualityField.helpTip = loc(322);	// Page preview JPEG quality
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.jpegQualityLabel = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.add('statictext', undefined, loc(323));	// quality
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.jpegQualityLabel.graphics.font = font;


						dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.alignChildren = ['left','middle'];
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFCheck = dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.add("checkbox", undefined, loc(50));
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFCheck.value = exportPagePDFs;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFCheck.helpTip = loc(169);	// Create a PDF of every single page
							dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown = dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.add('dropdownlist', undefined, '', {items:exportPDFpresetsNames});
								try { dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown.items[exportPagePDFusePresetIndex].selected = true; }
								catch(e) { dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown.items[0].selected = true; }
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown.maximumSize.width = dropDownMaxWidth1;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFCheck.helpTip = loc(170);	// PDF settings to use for single page PDFs
						dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.alignChildren = ['left','middle'];
							dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFCheck = dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.add("checkbox", undefined, loc(51));
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFCheck.value = exportDocumentPDF;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFCheck.helpTip = loc(171);	// Create a PDF of the entire document
							dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFspreadsCheck = dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.add("checkbox", undefined, loc(98));
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFspreadsCheck.value = exportDocumentPDFspreads;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFspreadsCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFspreadsCheck.helpTip = loc(172);	// Create doument PDF as spread
							dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown = dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.add('dropdownlist', undefined, '', {items:exportPDFpresetsNames});
								try { dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown.items[exportDocumentPDFusePresetIndex].selected = true; }
								catch(e) { dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown.items[0].selected = true; }
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown.maximumSize.width = dropDownMaxWidth1;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown.helpTip = loc(173);	// PDF settings to use for document PDFs


						dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup = dlgw.leftGroup.tabPanel.tab1.sub1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.orientation = 'column';
						dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.margins = [0,10,0,0];
							dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesLabel = dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.add('statictext', undefined, loc(180));	// PRO features title:
								dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesLabel.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesCheck = dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.add("checkbox", undefined, loc(97));	// export PRO attributes
								dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesCheck.value = exportPROFeatures;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesCheck.helpTip = loc(174);	// Export PRO features and attributes
							if (enableXMLtagsMenu) {
								dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.includeXMLtagsCheck = dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.add("checkbox", undefined, loc(178));	// export custom XML
									dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.includeXMLtagsCheck.value = includeXMLtags;
									dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.includeXMLtagsCheck.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.includeXMLtagsCheck.indent = 20;
									dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.includeXMLtagsCheck.helpTip = loc(179);	// Export custom XML
							}

					dlgw.leftGroup.tabPanel.tab1.sub2Group = dlgw.leftGroup.tabPanel.tab1.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab1.sub2Group.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab1.sub2Group.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab1.sub2Group.orientation = 'column';
					dlgw.leftGroup.tabPanel.tab1.sub2Group.spacing = 5;
					dlgw.leftGroup.tabPanel.tab1.sub2Group.margins = [25,0,0,0];
						dlgw.leftGroup.tabPanel.tab1.sub2Group.splittingLabel = dlgw.leftGroup.tabPanel.tab1.sub2Group.add('statictext', undefined, loc(54));	// Story splitting:
							dlgw.leftGroup.tabPanel.tab1.sub2Group.splittingLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab1.sub2Group.myNoSplitButton = dlgw.leftGroup.tabPanel.tab1.sub2Group.add('radiobutton', undefined, loc(55));	// Do NOT split stories
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myNoSplitButton.indent = 20;
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myNoSplitButton.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myNoSplitButton.value = (splittingStoriesOption==0);
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myNoSplitButton.helpTip = loc(175);	// Do NOT splite chained text boxes at all
						dlgw.leftGroup.tabPanel.tab1.sub2Group.myPageSplitButton = dlgw.leftGroup.tabPanel.tab1.sub2Group.add('radiobutton', undefined, loc(56));	// Split chained story frames per page
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myPageSplitButton.indent = 20;
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myPageSplitButton.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myPageSplitButton.value = (splittingStoriesOption==1);
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myPageSplitButton.helpTip = loc(176);	// Split chained text boxes when crossing the page boundary
						dlgw.leftGroup.tabPanel.tab1.sub2Group.myAllSplitButton = dlgw.leftGroup.tabPanel.tab1.sub2Group.add('radiobutton', undefined, loc(57));	// Split ALL chained story frames
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myAllSplitButton.indent = 20;
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myAllSplitButton.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myAllSplitButton.value = (splittingStoriesOption==2);
							dlgw.leftGroup.tabPanel.tab1.sub2Group.myAllSplitButton.helpTip = loc(177);	// Split ALL chained text boxes


				//------------------------------
				// images
				dlgw.leftGroup.tabPanel.tab2 = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(156));	// Images

					dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingLabel = dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.add('statictext', undefined, loc(370));	// images exports
							dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingLabel.graphics.font = font;
							/* with installed ImageMagick only
							var imageHandlingStringList = [loc(372),loc(373),loc(374),loc(375),loc(376)];	// set popup entries for handling missing images
							dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown = dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.add('dropdownlist', undefined, '', {items:imageHandlingStringList});
								dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown.maximumSize.width = dropDownMaxWidth2;
								dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown.items[imageMissingCreateFromPagePreview].selected = true;
								dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown.helpTip = loc(371);	// handling missing images 
              */

					dlgw.leftGroup.tabPanel.tab2.imageTypeGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.margins = [0,10,0,0];
					dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.alignChildren = ['left','middle'];
						//var exportImagesDropStringList = [loc(100),loc(101),loc(102),loc(103),loc(104)];	// set output formats None, JPEG, PNG, GIF, TIFF (when ImageMagick is installed)
						var exportImagesDropStringList = [loc(100),loc(101),loc(102)];	// set output formats None, JPEG, PNG (images export done by InDesign)
						dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.exportImagesDropdown = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.add('dropdownlist', undefined, '', {items:exportImagesDropStringList});	// image format
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.exportImagesDropdown.items[exportImages].selected = true;
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.exportImagesDropdown.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.exportImagesDropdown.helpTip = loc(181);	// Images export format - or don't export images

						dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imagesLabel = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.add('statictext', undefined, loc(45));	// Images,
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imagesLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.scaledByLabel = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.add('statictext', undefined, loc(48));	// scaled by
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.scaledByLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageParametersField = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.add('edittext', undefined, imageParameters);	// at width value
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageParametersField.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageParametersField.characters = 15;
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageParametersField.helpTip = loc(182);	// Image size and multiple images export

            /* used with ImageMagick only
						dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageFinishLabel = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.add('statictext', undefined, loc(73));	// image finishing label
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageFinishLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageOutputFinishParamField = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.add('edittext', undefined, imageOutputFinishParam);	// finishing parameter
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageOutputFinishParamField.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageOutputFinishParamField.characters = 20;
							dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageOutputFinishParamField.helpTip = loc(183);	// Image finishing parameters
						*/

					dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.indent = 50;
					dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPILabel = dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.add('statictext', undefined, loc(360));	// image output DPI
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPILabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPIField = dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.add('edittext', undefined, imageDPI);
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPIField.graphics.font = font;
							//dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPIField.characters = 10;
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPIField.size = editSize3chars;
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPIField.helpTip = loc(361);
						dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityLabel = dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.add('statictext', undefined, loc(362));	// image JPEG quality
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityField = dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.add('edittext', undefined, imageQUALITY);
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityField.graphics.font = font;
							//dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityField.characters = 4;
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityField.size = editSize3chars;
							dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityField.helpTip = loc(363);
/*
					dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamLabel = dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.add('statictext', undefined, loc(364));	// image input params
							dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamField = dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.add('edittext', undefined, imageInputParam);
							dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamField.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamField.characters = 30;
							dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamField.helpTip = loc(365);
*/
					dlgw.leftGroup.tabPanel.tab2.imageCopyGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.margins = [0,10,0,0];
					dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.alignChildren = ['left','middle'];
            /* used with ImageMagick only
						var copyOriginalImageFilesStringList = [loc(85),loc(86),loc(87)];	// set popup entries for: Copy original image files
						dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.copyOriginalImageFilesDropdown = dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.add('dropdownlist', undefined, '', {items:copyOriginalImageFilesStringList});	// image copy
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.copyOriginalImageFilesDropdown.items[copyOriginalImageFiles].selected = true;
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.copyOriginalImageFilesDropdown.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.copyOriginalImageFilesDropdown.helpTip = loc(184);	// Copy original images to output folder 
					  */

						dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDELabel = dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.add('statictext', undefined, loc(139));	// images to exclude when name is starting with one of these strings
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDELabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDEField = dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.add('edittext', undefined, imageEXCLUDE);
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDEField.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDEField.characters = 20;
							dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDEField.helpTip = loc(140);

					dlgw.leftGroup.tabPanel.tab2.imageMetaGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.margins = [0,10,0,0];
					dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.orientation = 'row';
/*
						dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.extractImageMetaDataCheck = dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.add("checkbox", undefined, loc(377));
							dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.extractImageMetaDataCheck.value = (extractImageMetaData > 0);
							dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.extractImageMetaDataCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.extractImageMetaDataCheck.helpTip = loc(378);	// add prefix

						dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.fullImageMetaDataCheck = dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.add("checkbox", undefined, loc(379));
							dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.fullImageMetaDataCheck.value = (fullImageMetaData > 0);
							dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.fullImageMetaDataCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.fullImageMetaDataCheck.helpTip = loc(380);	// add prefix
*/

					dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup = dlgw.leftGroup.tabPanel.tab2.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.orientation = 'column';
/*
						dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageAddTargetPrefixCheck = dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.add("checkbox", undefined, loc(368));
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageAddTargetPrefixCheck.value = (imageAddTargetPrefix > 0);
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageAddTargetPrefixCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageAddTargetPrefixCheck.helpTip = loc(369);	// add prefix
*/
						dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageOverwriteTargetIfExistsCheck = dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.add("checkbox", undefined, loc(366));
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageOverwriteTargetIfExistsCheck.value = (imageOverwriteTargetIfExists > 0);
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageOverwriteTargetIfExistsCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageOverwriteTargetIfExistsCheck.helpTip = loc(367);	// overwrite images
/*
						dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageDoROTATECheck = dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.add("checkbox", undefined, loc(99));
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageDoROTATECheck.value = (imageDoROTATE > 0);
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageDoROTATECheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageDoROTATECheck.helpTip = loc(185);	// rotate images
*/

				//------------------------------
				// Box catching
				dlgw.leftGroup.tabPanel.tab3 = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(157));	// Box Catching
					dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup = dlgw.leftGroup.tabPanel.tab3.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusLabel = dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.add('statictext', undefined, loc(90));	// Catch Radius
							dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusField = dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.add('edittext', undefined, catchRadius.toString());
							dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusField.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusField.characters = 4;
							dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusField.helpTip = loc(190);	// Catch radius
					dlgw.leftGroup.tabPanel.tab3.catchBoxGroup = dlgw.leftGroup.tabPanel.tab3.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.orientation = "column";
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.allBoxesNoCatchCheck = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.add("checkbox", undefined, loc(119));
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.allBoxesNoCatchCheck.value = (allBoxesNoCatch > 0);
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.allBoxesNoCatchCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.allBoxesNoCatchCheck.helpTip = loc(191);	// no catch at all
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2 = dlgw.leftGroup.tabPanel.tab3.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.alignChildren = ['left','top'];
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.orientation = "column";
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.spacing = 3;
						dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.margins = [18,0,0,0];
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.nocatchLabel = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.add('statictext', undefined, loc(120));	// no catch trough
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.nocatchLabel.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.unassignedBoxesNoCatchCheck = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.add("checkbox", undefined, loc(121));
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.unassignedBoxesNoCatchCheck.value = unassignedBoxesNoCatch;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.unassignedBoxesNoCatchCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.unassignedBoxesNoCatchCheck.helpTip = loc(192);	// unassigned content no catch
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.emptyBoxesNoCatchCheck = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.add("checkbox", undefined, loc(122));
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.emptyBoxesNoCatchCheck.value = emptyBoxesNoCatch;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.emptyBoxesNoCatchCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.emptyBoxesNoCatchCheck.helpTip = loc(193);	// empty boxes no catch
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.textBoxesNoCatchCheck = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.add("checkbox", undefined, loc(123));
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.textBoxesNoCatchCheck.value = textBoxesNoCatch;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.textBoxesNoCatchCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.textBoxesNoCatchCheck.helpTip = loc(194);	// text boxes no catch
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.chainedBoxesNoCatchCheck = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.add("checkbox", undefined, loc(124));
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.chainedBoxesNoCatchCheck.value = chainedBoxesNoCatch;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.chainedBoxesNoCatchCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.chainedBoxesNoCatchCheck.helpTip = loc(195);	// chained text boxes no catch
							dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.imageBoxesNoCatchCheck = dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.add("checkbox", undefined, loc(125));
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.imageBoxesNoCatchCheck.value = imageBoxesNoCatch;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.imageBoxesNoCatchCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.imageBoxesNoCatchCheck.helpTip = loc(196);	// Image boxes no catch


				//------------------------------
				// Excludes
				dlgw.leftGroup.tabPanel.tab4 = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(158));	// Excludes
				dlgw.leftGroup.tabPanel.tab4.orientation = 'row';
					dlgw.leftGroup.tabPanel.tab4.excludes1Group = dlgw.leftGroup.tabPanel.tab4.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab4.excludes1Group.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab4.excludes1Group.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.tab4.excludes1Group.orientation = 'column';
						dlgw.leftGroup.tabPanel.tab4.excludes1Group.excludeHiddenLayersCheck = dlgw.leftGroup.tabPanel.tab4.excludes1Group.add("checkbox", undefined, loc(105));
							dlgw.leftGroup.tabPanel.tab4.excludes1Group.excludeHiddenLayersCheck.value = (excludeHiddenLayers > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes1Group.excludeHiddenLayersCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes1Group.excludeHiddenLayersCheck.helpTip = loc(201);	// Don't export content on invisible layers

						dlgw.leftGroup.tabPanel.tab4.excludes1cGroup = dlgw.leftGroup.tabPanel.tab4.excludes1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.orientation = 'row';
							dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersLabel = dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.add('statictext', undefined, loc(337));
								dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersLabel.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersField = dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.add("edittext", undefined, excludeLayers);
								dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersField.size = editSize;
								dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersField.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersField.helpTip = loc(338);	// "A comma separated list of layer names to exclude from export"

						var spacer1 = dlgw.leftGroup.tabPanel.tab4.excludes1Group.add('statictext', undefined, ""); spacer1.size = [15,10];	// spacer
						// ---------------
						dlgw.leftGroup.tabPanel.tab4.excludes1aGroup = dlgw.leftGroup.tabPanel.tab4.excludes1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.orientation = 'row';
							dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.labelThreshold = dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.add('statictext', undefined, loc(330));
								dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.labelThreshold.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.pageframingBoxSizeThresholdField = dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.add('edittext', undefined, pageframingBoxSizeThreshold);
								dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.pageframingBoxSizeThresholdField.size = editSize3chars;
								dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.pageframingBoxSizeThresholdField.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.pageframingBoxSizeThresholdField.helpTip = loc(331);	// page framer threshold

						dlgw.leftGroup.tabPanel.tab4.excludes1bGroup = dlgw.leftGroup.tabPanel.tab4.excludes1Group.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.orientation = 'column';
							dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.labelThreshold = dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.add('statictext', undefined, loc(332));
								//dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.labelThreshold.size = labelSize;
								dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.labelThreshold.graphics.font = font;
							var excludeLargeBoxesStringList = [loc(333),loc(334),loc(335),loc(336)];
							var thresIDX = 0;
							switch (pageframingBoxNoExport) {
								case 0: break;	// export normal
								case 1: thresIDX = 1; break;	// don't export text boxes
								case 2: thresIDX = 2; break;	// don't export image boxes
								case 12: thresIDX = 3; break;	// don't export image boxes
							}
							dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown = dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.add('dropdownlist', undefined, '', {items:excludeLargeBoxesStringList});	// image copy
								dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown.items[thresIDX].selected = true;
								dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown.helpTip = loc(331);
								dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown.indent = indent;


					dlgw.leftGroup.tabPanel.tab4.excludes2Group = dlgw.leftGroup.tabPanel.tab4.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab4.excludes2Group.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab4.excludes2Group.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab4.excludes2Group.orientation = 'column';
					dlgw.leftGroup.tabPanel.tab4.excludes2Group.spacing = 3;
					dlgw.leftGroup.tabPanel.tab4.excludes2Group.margins = [30,0,0,0];
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.excludeBoxesLabel = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add('statictext', undefined, loc(106));	// Exclude Elements from Export:
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.excludeBoxesLabel.graphics.font = font;
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressAllEmptyBoxCheck = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add("checkbox", undefined, loc(107));
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressAllEmptyBoxCheck.value = (suppressAllEmptyBox > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressAllEmptyBoxCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressAllEmptyBoxCheck.helpTip = loc(202);	// Don't export all empty boxes
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyTextBoxCheck = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add("checkbox", undefined, loc(108));
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyTextBoxCheck.value = (suppressEmptyTextBox > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyTextBoxCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyTextBoxCheck.indent = indent;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyTextBoxCheck.helpTip = loc(203);	// Don't export all empty text boxes
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyImageBoxCheck = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add("checkbox", undefined, loc(109));
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyImageBoxCheck.value = (suppressEmptyImageBox > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyImageBoxCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyImageBoxCheck.indent = indent;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyImageBoxCheck.helpTip = loc(204);	// Don't export all empty image boxes
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressUnassignedBoxCheck = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add("checkbox", undefined, loc(110));
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressUnassignedBoxCheck.value = (suppressUnassignedBox > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressUnassignedBoxCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressUnassignedBoxCheck.indent = indent;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressUnassignedBoxCheck.helpTip = loc(205);	// Don't export boxes with unassigned content
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressLineBoxCheck = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add("checkbox", undefined, loc(111));
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressLineBoxCheck.value = (suppressLineBox > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressLineBoxCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressLineBoxCheck.indent = indent;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressLineBoxCheck.helpTip = loc(206);	// Don't export line boxes
						dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressNonPrintingBoxCheck = dlgw.leftGroup.tabPanel.tab4.excludes2Group.add("checkbox", undefined, loc(112));
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressNonPrintingBoxCheck.value = (suppressNonPrintingBox > 0 ? true : false);
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressNonPrintingBoxCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressNonPrintingBoxCheck.helpTip = loc(207);	// Don't export non printing boxes


				//------------------------------
				// tables
				dlgw.leftGroup.tabPanel.tab5 = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(159));	// Tables
					dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup = dlgw.leftGroup.tabPanel.tab5.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.tableOptionsLabel = dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.add('statictext', undefined, loc(59));	// Table Options
							dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.tableOptionsLabel.graphics.font = font;
						var tableCollapsList = [loc(60),loc(61),loc(62)];
						dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.TABLE_BORDER_COLLAPSEDropdown = dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.add('dropdownlist', undefined, '', {items:tableCollapsList});	// image format
							dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.TABLE_BORDER_COLLAPSEDropdown.items[parseInt(TABLE_BORDER_COLLAPSE)].selected = true;
							dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.TABLE_BORDER_COLLAPSEDropdown.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.TABLE_BORDER_COLLAPSEDropdown.helpTip = loc(211);	// How to treat table cell frames
						dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.cellFramesLabel = dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.add('statictext', undefined, loc(63));	// Cell frames
							dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.cellFramesLabel.graphics.font = font;
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup = dlgw.leftGroup.tabPanel.tab5.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.orientation = "row";
						dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.cellResizeLabel = dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.add('statictext', undefined, loc(64), {multiline:true});	// Cell resize factor
							dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.cellResizeLabel.graphics.font = font;
							//dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.cellResizeLabel.size = labelSize2;
						dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.TABLE_CELLS_WIDTH_PRESERVEField = dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.add('edittext', undefined, TABLE_CELLS_WIDTH_PRESERVE.toString());
							dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.TABLE_CELLS_WIDTH_PRESERVEField.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.TABLE_CELLS_WIDTH_PRESERVEField.size = editSize3chars;
							dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.TABLE_CELLS_WIDTH_PRESERVEField.helpTip = loc(212);	// Table cell sizes
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2 = dlgw.leftGroup.tabPanel.tab5.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.orientation = "row";
					dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.margins = [30,0,0,0];
						dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.cellHelpLabel = dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.add('statictext', undefined, loc(65));	//  0 = floating cell widths, 1.0 = retain, 1.3 = size by
							dlgw.leftGroup.tabPanel.tab5.cellResizeGroup2.cellHelpLabel.graphics.font = font;


				//------------------------------
				// browser view
				dlgw.leftGroup.tabPanel.tab6 = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(160));	// Browser View
				dlgw.leftGroup.tabPanel.tab6.orientation = "column";
					dlgw.leftGroup.tabPanel.tab6.top = dlgw.leftGroup.tabPanel.tab6.add('group', undefined);
					dlgw.leftGroup.tabPanel.tab6.top.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.tab6.top.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.tab6.top.orientation = "row";
						dlgw.leftGroup.tabPanel.tab6.top.leftGroup = dlgw.leftGroup.tabPanel.tab6.top.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab6.top.leftGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab6.top.leftGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.tab6.top.leftGroup.orientation = "column";
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.add('group', undefined);
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.alignment = ['left','top'];
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.alignChildren = ['left','middle'];
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameLabel = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.add('statictext', undefined, loc(66));	// CSS name
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameLabel.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameField = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.add('edittext', undefined, cssName);
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameField.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameField.size = editSize2;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameField.helpTip = loc(221);	// CSS name
	
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.CSSclassNamesASCIICheck = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.add("checkbox", undefined, loc(88));	// Allow UTF-8 in CSS class names - or plain ASCII only
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.CSSclassNamesASCIICheck.value = (CSSclassNamesASCII > 0 ? true : false);
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.CSSclassNamesASCIICheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.CSSclassNamesASCIICheck.helpTip = loc(222);	// ASCII CSS class names only
	
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.add('group', undefined);
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.alignment = ['left','top'];
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.alignChildren = ['left','middle'];
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorLabel = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.add('statictext', undefined, loc(80));	// Font size factor
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorLabel.graphics.font = font;
								var fontszi = parseInt(fontsizeFactor).toString();
								var fontszf = parseFloat(fontsizeFactor).toString();
								if (fontszi == fontszf) fontszf += ".0";
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorField = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.add('edittext', undefined, fontszf);
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorField.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorField.size = editSize5chars;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorField.helpTip = loc(228);	// fontsizeFactor
	/****** in test phase
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.add('group', undefined);
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.alignment = ['left','top'];
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.alignChildren = ['left','middle'];
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsLabel = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.add('statictext', undefined, loc(340));	// Font units pt % em
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsLabel.graphics.font = font;
								var fontsizeUnitsStringList = [loc(341),loc(342),loc(343)];
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsDropdown = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.add('dropdownlist', undefined, '', {items:fontsizeUnitsStringList});
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsDropdown.items[fontsizeUnits].selected = true;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsDropdown.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsDropdown.helpTip = loc(340);
	
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.add('group', undefined);
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.alignment = ['left','top'];
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.alignChildren = ['left','middle'];
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseLabel = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.add('statictext', undefined, loc(350));	// Font size base
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseLabel.graphics.font = font;
								var fontszi = parseInt(fontsizeBase).toString();
								var fontszf = parseFloat(fontsizeBase).toString();
								if (fontszi == fontszf) fontszf += ".0";
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseField = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.add('edittext', undefined, fontszf);
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseField.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseField.characters = 6;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseField.helpTip = loc(351);	// fontsizeBase
	*****/
							//**** magnifying: copy original page JPEG exported from InDesign
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.add('group', undefined);
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.alignment = ['left','top'];
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.alignChildren = ['left','middle'];
							dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.margins = [0,15,0,15];
								var viewModeList = [loc(76),loc(77)];
								dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.viewModeDropdown = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.add('dropdownlist', undefined, '', {items:viewModeList});	// view mode list
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.viewModeDropdown.items[magnifyingGlass].selected = true;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.viewModeDropdown.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.viewModeDropdown.helpTip = loc(75);	// View mode when navigating articles on a page


						dlgw.leftGroup.tabPanel.tab6.top.rightGroup = dlgw.leftGroup.tabPanel.tab6.top.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab6.top.rightGroup.alignChildren = ['left','top'];
						dlgw.leftGroup.tabPanel.tab6.top.rightGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab6.top.rightGroup.margins = [30,0,0,0];
						dlgw.leftGroup.tabPanel.tab6.top.rightGroup.orientation = "column";
						dlgw.leftGroup.tabPanel.tab6.top.rightGroup.spacing = 5;
							dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressMouseOverLabel = dlgw.leftGroup.tabPanel.tab6.top.rightGroup.add('statictext', undefined, loc(130));	// CSS name
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressMouseOverLabel.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressAllMouseOversCheck = dlgw.leftGroup.tabPanel.tab6.top.rightGroup.add("checkbox", undefined, loc(131));
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressAllMouseOversCheck.value = (suppressExportMouseOvers.indexOf("1") >= 0 ? true : false);
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressAllMouseOversCheck.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressAllMouseOversCheck.helpTip = loc(224);	// Suppress all mouse over articles
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressTextMouseOversCheck = dlgw.leftGroup.tabPanel.tab6.top.rightGroup.add("checkbox", undefined, loc(132));
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressTextMouseOversCheck.value = (suppressExportMouseOvers.indexOf("2") >= 0 ? true : false);
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressTextMouseOversCheck.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressTextMouseOversCheck.indent = indent;
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressTextMouseOversCheck.helpTip = loc(225);	// Suppress mouse over text boxes
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressImageMouseOversCheck = dlgw.leftGroup.tabPanel.tab6.top.rightGroup.add("checkbox", undefined, loc(133));
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressImageMouseOversCheck.value = (suppressExportMouseOvers.indexOf("4") >= 0 ? true : false);
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressImageMouseOversCheck.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressImageMouseOversCheck.indent = indent;
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressImageMouseOversCheck.helpTip = loc(226);	// Suppress mouse over image boxes
								dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressEmptyMouseOversCheck = dlgw.leftGroup.tabPanel.tab6.top.rightGroup.add("checkbox", undefined, loc(134));
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressEmptyMouseOversCheck.value = (suppressExportMouseOvers.indexOf("8") >= 0 ? true : false);
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressEmptyMouseOversCheck.graphics.font = font;
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressEmptyMouseOversCheck.indent = indent;
									dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressEmptyMouseOversCheck.helpTip = loc(227);	// suppress mouse over empty boxes
	
					dlgw.leftGroup.tabPanel.tab6.bottom = dlgw.leftGroup.tabPanel.tab6.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab6.bottom.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab6.bottom.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.tab6.bottom.orientation = "row";
						dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup = dlgw.leftGroup.tabPanel.tab6.bottom.add('group', undefined);
						dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.alignChildren = ['left','top'];
						dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.orientation = "column";
							dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsLabel = dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.add('statictext', undefined, loc(390));	// websiteParams
								dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsLabel.graphics.font = font;
							dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsField = dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.add('edittext', undefined, websiteParams, {multiline:true});
								dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsField.graphics.font = font;
								dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsField.size = [530,55]
								dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsField.characters = 6;
								dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsField.helpTip = loc(391);


				//------------------------------
				// paths info tab
				dlgw.leftGroup.tabPanel.pathInfosTab = dlgw.leftGroup.tabPanel.add('tab', undefined, loc(161));	// Path Infos
				dlgw.leftGroup.tabPanel.pathInfosTab.spacing = 1;
				dlgw.leftGroup.tabPanel.pathInfosTab.margins = [5,5,0,0];
					// INIT file
					if (initFileNameLoaded != "") {
						dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
						dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.alignChildren = ['left','top'];
						dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.alignment = ['left','top'];
							dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.label = dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.add('statictext', undefined, loc(83), {multiline:false});	// init File:
								dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.label.size = pathInfosSize;
								dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.label.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.add('statictext', undefined, initFileNameLoaded);
								dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.path.size = pathInfosSize2;
								dlgw.leftGroup.tabPanel.pathInfosTab.initFileGroup.path.graphics.font = font;
					}

					// settings file
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.alignChildren = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.margins = [0,0,0,0];
						dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.chooseSettingsPathBtn = dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.add('button', undefined, loc(74));
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.chooseSettingsPathBtn.helpTip = loc(310);	// Choose new path for settings files and communication
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.chooseSettingsPathBtn.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.chooseSettingsPathBtn.size = pathInfosButtonSize;
						dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.add('statictext', undefined, dotText(settingsfldr.fsName,dotdotWidth));
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.path.size = pathInfosSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.path.graphics.font = font;
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.margins = [0,0,0,0];
						dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.label = dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.add('statictext', undefined, loc(89), {multiline:false});	// Settings File:
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.label.size = pathInfosSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.label.justify = "right";
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.label.graphics.font = font;
						dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.add('statictext', undefined, settingsFileName);
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.path.graphics.font = font;
/*
					if (settingsloaded != 0) {
						dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.message = dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.add('statictext', undefined, loc(84), {multiline:false});	// Settings File not reachable
							//dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.message.size = pathInfosSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.message.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.settingsFileGroup.message.graphics.foregroundColor = redpen;
					}
*/

					// Hooks path
					dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.margins = [0,5,0,0];
					dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.choosehooksPathBtn = dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.add('button', undefined, loc(308));
							dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.choosehooksPathBtn.helpTip = loc(309);	// Choose hooks base path
							dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.choosehooksPathBtn.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.choosehooksPathBtn.size = pathInfosButtonSize;
						dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.add('edittext', undefined, decodeURI(hooksBasePath));
							dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.path.size = pathEditSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.path.graphics.font = font;

					// Export path
					dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.margins = [0,5,0,0];
					dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.chooseMainExportFolderBtn = dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.add('button', undefined, loc(30));
							dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.chooseMainExportFolderBtn.helpTip = loc(311);	// Choose main export base path
							dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.chooseMainExportFolderBtn.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.chooseMainExportFolderBtn.size = pathInfosButtonSize;
						dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.add('edittext', undefined, mainExpfldr.fsName);
							dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.path.size = pathEditSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.path.graphics.font = font;
					// in and out paths seen from this script
					dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.label = dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.add('statictext', undefined, loc(41), {multiline:false});	// Source Export Path:
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.label.size = pathInfosSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.label.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.label.justify = "right";
						dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.add('statictext', undefined, dotText(infldr.fsName,dotdotWidth));
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.path.size = pathInfosSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.path.graphics.font = font;
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.margins = [0,0,0,0];
						dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.label = dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.add('statictext', undefined, loc(42), {multiline:false});	// XML Export Path:
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.label.size = pathInfosSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.label.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.label.justify = "right";
						dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.add('statictext', undefined, dotText(outfldr.fsName,dotdotWidth));
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.path.size = pathInfosSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.path.graphics.font = font;
					// in and out paths seen from the transformer - it may run on a different system and does not see the paths like we do: windows - OSX paths
					/* -----currently disabled
					dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.alignChildren = ['left','middle'];
						dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.label = dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.add('statictext', undefined, loc(316), {multiline:false});	// Source Export Path seen from transformer:
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.label.size = pathInfosSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.label.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.label.justify = "right";
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.label.helpTip = loc(317);	// set transformer specific path
						dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.path = dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.add('edittext', undefined, mainExportFolderInTransformer);
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.path.size = pathEditSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.path.graphics.font = font;
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.margins = [0,0,0,0];
						dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.label = dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.add('statictext', undefined, loc(318), {multiline:false});	// XML Export Path seen from transformer:
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.label.size = pathInfosSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.label.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.label.justify = "right";
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.label.helpTip = loc(319);	// set transformer specific path
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.label.margins = [0,0,0,0];
						dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.path = dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.add('edittext', undefined, mainExportFolderOutTransformer);
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.path.size = pathEditSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.path.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.path.margins = [0,0,0,0];
					*/
					// start the transformer
					dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.margins = [0,5,0,0];
						dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.startTransformerCheck = dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.add("checkbox", undefined, loc(43));
							dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.startTransformerCheck.value = doStartBatchXSLT;
							dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.startTransformerCheck.size = pathInfosCheckSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.startTransformerCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.startTransformerCheck.helpTip = loc(312);	// start transformer
						dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.path = dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.add('statictext', undefined, dotText(trans.fsName,dotdotWidth));
							dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.path.size = pathInfosSize2;
							dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.path.graphics.font = font;

					dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup = dlgw.leftGroup.tabPanel.pathInfosTab.add('group', undefined);
					dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.alignment = ['left','top'];
					dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.alignChildren = ['left','middle'];
					dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.margins = [0,5,0,0];
					dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.orientation = 'column';
					dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.spacing = 2;
						dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.controlTransformerCheck = dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.add("checkbox", undefined, loc(314));
							dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.controlTransformerCheck.value = doControlBatchXSLT;
							dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.controlTransformerCheck.size = pathInfosCheckSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.controlTransformerCheck.graphics.font = font;
							dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.controlTransformerCheck.helpTip = loc(315);	// control transformer

						dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1 = dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.add('group', undefined);
						dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.alignment = ['left','top'];
						dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.alignChildren = ['left','middle'];
							dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.chooseCommunicationFolderBtn = dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.add('button', undefined, loc(260));
								dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.chooseCommunicationFolderBtn.helpTip = loc(313);	// Choose new path for communicating with tranformer
								dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.chooseCommunicationFolderBtn.graphics.font = font;
								dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.chooseCommunicationFolderBtn.size = pathInfosButtonSize;
							dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.path = dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.add('statictext', undefined, dotText(commfldr.fsName,dotdotWidth));
								dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.path.size = pathInfosSize2;
								dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.path.graphics.font = font;



			//------------------------------
			// actions after export
			dlgw.leftGroup.actionsGroup = dlgw.leftGroup.add('group', undefined);
			dlgw.leftGroup.actionsGroup.alignChildren = ['left','top'];
			dlgw.leftGroup.actionsGroup.alignment = ['left','top'];
				dlgw.leftGroup.actionsGroup.closeDocAfterDoneCheck = dlgw.leftGroup.actionsGroup.add("checkbox", undefined, loc(67));
					dlgw.leftGroup.actionsGroup.closeDocAfterDoneCheck.value = closeDocumentAfterDone;
					dlgw.leftGroup.actionsGroup.closeDocAfterDoneCheck.graphics.font = font;
					dlgw.leftGroup.actionsGroup.closeDocAfterDoneCheck.helpTip = loc(231);	// Close document after base export is done
				dlgw.leftGroup.actionsGroup.showExportFolderCheck = dlgw.leftGroup.actionsGroup.add("checkbox", undefined, loc(69));
					dlgw.leftGroup.actionsGroup.showExportFolderCheck.value = showExportFolder;
					dlgw.leftGroup.actionsGroup.showExportFolderCheck.graphics.font = font;
					dlgw.leftGroup.actionsGroup.showExportFolderCheck.helpTip = loc(232);	// Show the export folder after base export is done
				dlgw.leftGroup.actionsGroup.showMessageCheck = dlgw.leftGroup.actionsGroup.add("checkbox", undefined, loc(72));
					dlgw.leftGroup.actionsGroup.showMessageCheck.value = showMessage;
					dlgw.leftGroup.actionsGroup.showMessageCheck.graphics.font = font;
					dlgw.leftGroup.actionsGroup.showMessageCheck.helpTip = loc(233);	// Show status messages during base export
				dlgw.leftGroup.actionsGroup.saveSettingsCheck = dlgw.leftGroup.actionsGroup.add("checkbox", undefined, loc(68));
					dlgw.leftGroup.actionsGroup.saveSettingsCheck.value = saveSettings;
					dlgw.leftGroup.actionsGroup.saveSettingsCheck.graphics.font = font;
					dlgw.leftGroup.actionsGroup.saveSettingsCheck.helpTip = loc(234);	// Save the current settings into the settings file


		//------------------------------
		// and the buttons on right side
		dlgw.rightGroup = dlgw.add('group');
		dlgw.rightGroup.orientation = 'column';
		dlgw.rightGroup.alignChildren = ['fill','top'];
		dlgw.rightGroup.alignment = ['fill','top'];
			dlgw.rightGroup.btnPanel = dlgw.rightGroup.add('panel', undefined, '');
				dlgw.rightGroup.btnPanel.exportBtn = dlgw.rightGroup.btnPanel.add('button', undefined, loc(141), {name:'ok'});
				dlgw.rightGroup.btnPanel.exportBtn.helpTip = loc(241);	// Start Export with these settings
				dlgw.rightGroup.btnPanel.cancelBtn = dlgw.rightGroup.btnPanel.add('button', undefined, loc(142), {name:'cancel'});
				dlgw.rightGroup.btnPanel.cancelBtn.helpTip = loc(242);	// Cancel Export

			dlgw.rightGroup.sub2Group = dlgw.rightGroup.add('group');
			dlgw.rightGroup.sub2Group.orientation = 'column';
			dlgw.rightGroup.sub2Group.alignChildren = ['fill','top'];
			dlgw.rightGroup.sub2Group.alignment = ['fill','top'];
			dlgw.rightGroup.sub2Group.margins = [0,62,0,0];
				dlgw.rightGroup.sub2Group.loadSettingsFromFileBtn = dlgw.rightGroup.sub2Group.add('button', undefined, loc(143));
				dlgw.rightGroup.sub2Group.loadSettingsFromFileBtn.helpTip = loc(243);	// Load new settings from settings file
				dlgw.rightGroup.sub2Group.saveSettingsToFileBtn = dlgw.rightGroup.sub2Group.add('button', undefined, loc(144));
				dlgw.rightGroup.sub2Group.saveSettingsToFileBtn.helpTip = loc(244);	// save settings to file
				dlgw.rightGroup.sub2Group.factorySettingsBtn = dlgw.rightGroup.sub2Group.add('button', undefined, loc(145));
				dlgw.rightGroup.sub2Group.factorySettingsBtn.helpTip = loc(245);	// revert to factory settings


		/*
			The dialog item handlers
		*/
		// Export and Cancel buttons
		dlgw.rightGroup.btnPanel.cancelBtn.onClick = function() { dlgw.close(0); } 
		dlgw.rightGroup.btnPanel.exportBtn.onClick = function() { dlgw.close(1); } 
		// Settings buttons
		dlgw.rightGroup.sub2Group.loadSettingsFromFileBtn.onClick = function() {
			var f = new Folder(settingsFilePath);
			var selected = f.openDlg(loc(251));
			if (selected != null) {
				var fnew = new Folder(selected);
				settingsFilePath = fnew.path;
				settingsFileName = fnew.name;
				//alert("selected: " + selected + "\nfullpath: " + fnew.fsName + "\npath: " + fnew.path + "\nname: " + fnew.name);
				var loaded = loadSettingsFromFile(),
					currentDocument = app.activeDocument,
					currentDocumentName = currentDocument.name,
					currentDocumentPath = "";
				try { currentDocumentPath = currentDocument.filePath.toString(); } catch(e){}	// unsafed documents have no path
				if (currentDocumentFilePathName != "") {
					var tdf = new File(currentDocumentFilePathName);
					currentDocumentPath = tdf.path;
				}
				else {
					try {
						currentDocumentPath = currentDocument.filePath.toString();
					} catch (e) {
					}
				}
				if (doingBookExport) {
					retrieve_issue_year_date(currentBookName,currentBookPath);
					objectShortcut = retrieve_objectShortcut(currentBookName);
				}
				else {
					retrieve_issue_year_date(currentDocumentName,currentDocumentPath);
					objectShortcut = retrieve_objectShortcut(currentDocumentName);
				}
				build_export_paths("","loadSettingsFromFileBtn");
				dlgw.close(2);
			}
		} 
		dlgw.rightGroup.sub2Group.saveSettingsToFileBtn.onClick = function() {
			var savepath = settingsFilePath;
			if (endsWith(savepath,folderSeparator) == false) savepath += folderSeparator;
			savepath += settingsFileName;
			var f = new File(savepath),
				selected = f.saveDlg(loc(253));
			if (selected != null) {
				settingsFilePath = selected.path;
				settingsFileName = selected.name;
				settingsloaded = 0;
				if (endsWith(settingsFileName.toLowerCase(),".set") == false) settingsFileName += ".set";
				//alert("savepath: " + savepath + "\nselected: " + selected + "\nsettingsFilePath: " + settingsFilePath + "\nsettingsFileName: " + settingsFileName);
				dlgw.close(3);
			}
		} 
		dlgw.rightGroup.sub2Group.factorySettingsBtn.onClick = function() {
			var factoryreset = confirm(loc(252),true);	// Are you sure to revert to default factory settings?
			if (factoryreset == true) {
				factorySettings();
				dlgw.close(4);
			}
		}

		dlgw.leftGroup.companyNameGroup.companyNameField.onChanging = function() { updateDialogExportPaths(); };
		dlgw.leftGroup.objectNameGroup.objectNameField.onChanging = function() { updateDialogExportPaths(); };
		dlgw.leftGroup.objectShortcutGroup.objectShortcutField.onChanging = function() { updateDialogExportPaths(); };
		dlgw.leftGroup.issueYearGroup.issueYearField.onChanging = function() { 
				if (dlgw.leftGroup.issueYearGroup.issueYearField.text.length > 4) dlgw.leftGroup.issueYearGroup.issueYearField.text = dlgw.leftGroup.issueYearGroup.issueYearField.text.substr(0,4);
				updateDialogExportPaths();
			};
		dlgw.leftGroup.issueDateGroup.issueDateField.onChanging = function() {
				if (dlgw.leftGroup.issueDateGroup.issueDateField.text.length > 4) dlgw.leftGroup.issueDateGroup.issueDateField.text = dlgw.leftGroup.issueDateGroup.issueDateField.text.substr(0,8);
				updateDialogExportPaths();
			};
		dlgw.leftGroup.issueDateGroup.issueDateFormatDropdown.onChange = function() {
				fileDateFormat = dlgw.leftGroup.issueDateGroup.issueDateFormatDropdown.selection.text;
				var curdocName = app.activeDocument.name;
				var newdate = stringToDate(unescape(curdocName), fileDateFormat);
				dlgw.leftGroup.issueYearGroup.issueYearField.text = newdate.split(",")[0];
				dlgw.leftGroup.issueDateGroup.issueDateField.text = newdate.split(",")[1];
			};

		// Choose settings path button
		dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.chooseSettingsPathBtn.onClick = function() {
			var f = new Folder(settingsFilePath),
				fnew,
				selected = f.selectDlg(loc(310));
			if (selected != null) {
				fnew = new Folder(selected);
				settingsFilePath = fnew.fsName;
				//alert("selected: " + selected + "\nfullpath: " + fnew.fsName + "\npath: " + fnew.path + "\nname: " + fnew.name);
				dlgw.leftGroup.tabPanel.pathInfosTab.settingsPathGroup.path = settingsFilePath;
				dlgw.close(2);
			}
		} 

		// Choose hooks path path button
		dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.choosehooksPathBtn.onChanging = function() {
			var f = new Folder(hooksBasePath),
				fnew,
				selected = f.selectDlg(loc(307));
			if (selected != null) {
				fnew = new Folder(selected);
				hooksBasePath = fnew.fsName;
				dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.path.text = hooksBasePath;
			}
		}
		dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.path.onChanging = function() {
				hooksBasePath = dlgw.leftGroup.tabPanel.pathInfosTab.hooksPathGroup.path.text;
		}

		// Choose mainExport path button
		dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.chooseMainExportFolderBtn.onClick = function() {
			var f = new Folder(mainExportFolder),
				fnew,
				selected = f.selectDlg(loc(311));
			if (selected != null) {
				fnew = new Folder(selected);
				mainExportFolder = fnew.fsName;
				build_mainExportFolders();
				//alert("selected: " + selected + "\nfullpath: " + fnew.fsName + "\npath: " + fnew.path + "\nname: " + fnew.name);
				dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.path.text = mainExportFolder;
				dlgw.close(2);
			}
		}
		// edit main export path
		dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.path.onChanging = function() {
				var newMainExport = dlgw.leftGroup.tabPanel.pathInfosTab.mainExportPathGroup.path.text,
					fnew = new Folder(newMainExport);
				mainExportFolder = fnew.fsName;
				mainExportFolderInConst = mainExportFolder + folderSeparator + "in";
				mainExportFolderIn = mainExportFolderInConst;
				mainExportFolderOutConst = mainExportFolder + folderSeparator + "out";
				mainExportFolderOut = mainExportFolderOutConst;
				mainExportFolderDoneConst = mainExportFolder + folderSeparator + "done";
				mainExportFolderDone = mainExportFolderDoneConst;
				dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.path.text = mainExportFolderInConst;
				dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.path.text = mainExportFolderOutConst;
				build_mainExportFolders();
				//alert("changed: " + mainExportFolder);
		}

		// Choose communication path button
		dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.group1.chooseCommunicationFolderBtn.onClick = function() {
			var f = new Folder(commfldr.fsName),
				fnew,
				selected = f.selectDlg(loc(313));
			if (selected != null) {
				fnew = new Folder(selected);
				BXSLT4InDesignCommDir = fnew.fsName;
				build_mainExportFolders();
				//alert("selected: " + selected + "\nfullpath: " + fnew.fsName + "\npath: " + fnew.path + "\nname: " + fnew.name);
				dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.path.text = BXSLT4InDesignCommDir;
			}
		} 

		function updateDialogExportPaths() {
			companyName = dlgw.leftGroup.companyNameGroup.companyNameField.text;
			objectName = dlgw.leftGroup.objectNameGroup.objectNameField.text;
			objectShortcut = dlgw.leftGroup.objectShortcutGroup.objectShortcutField.text;
			issueYear = dlgw.leftGroup.issueYearGroup.issueYearField.text;
			issueDate = dlgw.leftGroup.issueDateGroup.issueDateField.text;
			build_export_paths("","mainDialog");
			//alert(dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.path.text);
			dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.path.text = dotText(new Folder(mainExportFolderInConst).fsName,dotdotWidth); //dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroup.add('statictext', undefined, dotText(infldr.fsName,dotdotWidth));
			dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.path.text = dotText(new Folder(mainExportFolderOutConst).fsName,dotdotWidth); //dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroup.add('statictext', undefined, dotText(outfldr.fsName,dotdotWidth));
		}

		// MUST set some options when changing the output mode
		dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown.onChange = 
			dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.onClick = function() {
				var outputmodeSelectIdx = dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown.selection.index;
				// MUST have page JPEGs for ePaper
				var pagejpegCheckvalue = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.value;
				if ((outputmodeSelectIdx <= 1) || (outputmodeSelectIdx == 5)) {	// for flipbook, pages epaper and slidebook
					dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.value = true;
					dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.enabled = false;
				}
				else dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.enabled = true;
				// MUST NOT have lupe for slidebook
				if (outputmodeSelectIdx == 5) {	// for slidebook
					dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.viewModeDropdown.items[0].selected = true;
				}
			};


	// before show main dialog: run hook
	runHooks(scriptName,"beforeShowMainDialog", null);

	// Display the dialog box.
	var go = dlgw.show();
	//alert(go);
	if ((go == 2) || (go == 4)) return(go);	// revert or load default settings

	// for debug only:
	//listProperties(dlgw);

	// Get export path fields
	companyName = dlgw.leftGroup.companyNameGroup.companyNameField.text;
	objectName = dlgw.leftGroup.objectNameGroup.objectNameField.text;
	objectShortcut = dlgw.leftGroup.objectShortcutGroup.objectShortcutField.text;
	issueYear = dlgw.leftGroup.issueYearGroup.issueYearField.text;
	issueDate = dlgw.leftGroup.issueDateGroup.issueDateField.text;

	//----------- tab1 fields: Output Types
	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown != null) {
		outputMode = dlgw.leftGroup.tabPanel.tab1.sub1Group.outputModeGroup.outputModeDropdown.selection.index;
		outputModeCustomXSL = availableOutputXSLs[outputMode];
		//alert(outputMode + "\n" +outputModeCustomXSL+ "\n\n" +availableOutputXSLs);
	}
	//Get page turn mode.
	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.pageTurnModeRadioGroup.turnRadio.value == true) {			// turning pages mode
		pageTurnMode = 'turn';
	}
	else pageTurnMode = 'slide';

	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup.exportPageJPEGCheck.value == true) exportPageJPEGs = true;
	else exportPageJPEGs = false;
	pageImageWidth = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageWidthField.text;
	//if (pageImageWidth.indexOf("x") < 0) pageImageWidth += "x";

	var pageImageQualitytmp = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageQualityField.text;
	if (!isNaN(parseInt(pageImageQualitytmp)) ) pageImageQuality = "" + parseInt(pageImageQualitytmp);
	if (pageImageQuality < 1) pageImageQuality = "1";
	if (pageImageQuality > 100) pageImageQuality = "100";

	var pageImageDPItmp = dlgw.leftGroup.tabPanel.tab1.sub1Group.pageJpegsGroup1.pageImageDPIField.text;
	if (!isNaN(pageImageDPItmp) ) pageImageDPI = pageImageDPItmp;

	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFCheck.value == true) exportPagePDFs = true;
	else exportPagePDFs = false;
	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown != null) {
		exportPagePDFusePresetIndex = dlgw.leftGroup.tabPanel.tab1.sub1Group.pagePDFsGroup.exportPagePDFpresetsDropdown.selection.index;
		exportPagePDFusePresetName = exportPDFpresetsNames[exportPagePDFusePresetIndex];
	}

	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFCheck.value == true) exportDocumentPDF = true;
	else exportDocumentPDF = false;
	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFspreadsCheck.value == true) exportDocumentPDFspreads = true;
	else exportDocumentPDFspreads = false;
	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown != null) {
		exportDocumentPDFusePresetIndex = dlgw.leftGroup.tabPanel.tab1.sub1Group.documentPDFsGroup.exportDocumentPDFpresetsDropdown.selection.index;
		exportDocumentPDFusePresetName = exportPDFpresetsNames[exportDocumentPDFusePresetIndex];
	}

	if (dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.exportPROFeaturesCheck.value == true) exportPROFeatures = true;
	else exportPROFeatures = false;
	if (enableXMLtagsMenu) {
		if (dlgw.leftGroup.tabPanel.tab1.sub1Group.attributesGroup.includeXMLtagsCheck.value == true) includeXMLtags = 1;
		else includeXMLtags = 0;
	}
	//Get the story splitting option.
	if (dlgw.leftGroup.tabPanel.tab1.sub2Group.myNoSplitButton.value == true) {			// no story splitting
		splittingStoriesOption = 0;
	}
	else if (dlgw.leftGroup.tabPanel.tab1.sub2Group.myPageSplitButton.value == true) {	// split stories per page
		splittingStoriesOption = 1;
	}
	else {												// split all story frames
		splittingStoriesOption = 2;
	}

	//----------- tab2 fields: Images
	/* only with ImageMagick installed
	if (dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown != null) {
		imageMissingCreateFromPagePreview = dlgw.leftGroup.tabPanel.tab2.imageHandlingGroup.imageHandlingDropdown.selection.index;
	}
	else imageMissingCreateFromPagePreview = 1;	// from original, missing images from image box
	*/

	if (dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.exportImagesDropdown != null) {
		exportImages = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.exportImagesDropdown.selection.index;
	}
	else exportImages = 1;	// default to JPEG

	/*imageInputParam = dlgw.leftGroup.tabPanel.tab2.imageParamsInputGroup.imageInputParamField.text;*/
	imageDPI = dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageDPIField.text;
	if (imageInputDPI < imageDPI) imageInputDPI = imageDPI;
	imageQUALITY = dlgw.leftGroup.tabPanel.tab2.imageParamsOutputGroup.imageQualityField.text;

	imageParameters = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageParametersField.text;
	/* (with ImageMagick only)
	imageOutputFinishParam = dlgw.leftGroup.tabPanel.tab2.imageTypeGroup.imageOutputFinishParamField.text;
	if (dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.copyOriginalImageFilesDropdown != null) {
		copyOriginalImageFiles = dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.copyOriginalImageFilesDropdown.selection.index;
	}
	*/
	
	imageEXCLUDE = dlgw.leftGroup.tabPanel.tab2.imageCopyGroup.imageEXCLUDEField.text;

/*
	if (dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.extractImageMetaDataCheck.value == true) extractImageMetaData = 1;
	else extractImageMetaData = 0;
	if (dlgw.leftGroup.tabPanel.tab2.imageMetaGroup.fullImageMetaDataCheck.value == true) fullImageMetaData = 1;
	else fullImageMetaData = 0;
*/
/*
	if (dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageAddTargetPrefixCheck.value == true) imageAddTargetPrefix = 1;
	else imageAddTargetPrefix = 0;
*/
	if (dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageOverwriteTargetIfExistsCheck.value == true) imageOverwriteTargetIfExists = 1;
	else imageOverwriteTargetIfExists = 0;
/*
	if (dlgw.leftGroup.tabPanel.tab2.imageOptionsGroup.imageDoROTATECheck.value == true) imageDoROTATE = 1;
	else imageDoROTATE = 0;
*/
	//----------- tab3 fields: Catching
	catchRadius = dlgw.leftGroup.tabPanel.tab3.catchRadiusGroup.catchRadiusField.text;
	try { catchRadius = Math.round(catchRadius); }
	catch(e) { catchRadius = 0; }
	if (isNaN(catchRadius) ) catchRadius = 0;
	if (dlgw.leftGroup.tabPanel.tab3.catchBoxGroup.allBoxesNoCatchCheck.value == true) allBoxesNoCatch = 1;
	else allBoxesNoCatch = 0;
	if (dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.unassignedBoxesNoCatchCheck.value == true) unassignedBoxesNoCatch = true;
	else unassignedBoxesNoCatch = false;
	if (dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.emptyBoxesNoCatchCheck.value == true) emptyBoxesNoCatch = true;
	else emptyBoxesNoCatch = false;
	if (dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.chainedBoxesNoCatchCheck.value == true) chainedBoxesNoCatch = true;
	else chainedBoxesNoCatch = false;
	if (dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.textBoxesNoCatchCheck.value == true) textBoxesNoCatch = true;
	else textBoxesNoCatch = false;
	if (dlgw.leftGroup.tabPanel.tab3.catchBoxGroup2.imageBoxesNoCatchCheck.value == true) imageBoxesNoCatch = true;
	else imageBoxesNoCatch = false;

	//----------- tab4 fields: Excludes
	if (dlgw.leftGroup.tabPanel.tab4.excludes1Group.excludeHiddenLayersCheck.value == true) excludeHiddenLayers = 1;
	else excludeHiddenLayers = 0;

	excludeLayers = dlgw.leftGroup.tabPanel.tab4.excludes1cGroup.excludeLayersField.text;

	var pageframingBoxSizeThresholdtmp = dlgw.leftGroup.tabPanel.tab4.excludes1aGroup.pageframingBoxSizeThresholdField.text;
	if (!isNaN(parseFloat(pageframingBoxSizeThresholdtmp)) ) pageframingBoxSizeThreshold = "" + parseFloat(pageframingBoxSizeThresholdtmp);
	if (dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown != null) {
		switch (dlgw.leftGroup.tabPanel.tab4.excludes1bGroup.excludePageFramingBoxesDropdown.selection.index) {
			case 0: pageframingBoxNoExport = 0; break;
			case 1: pageframingBoxNoExport = 1; break;
			case 2: pageframingBoxNoExport = 2; break;
			case 3: pageframingBoxNoExport = 12; break;
		}
	}

	if (dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressAllEmptyBoxCheck.value == true) suppressAllEmptyBox = 1;
	else suppressAllEmptyBox = 0;
	if (dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyTextBoxCheck.value == true) suppressEmptyTextBox = 1;
	else suppressEmptyTextBox = 0;
	if (dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressEmptyImageBoxCheck.value == true) suppressEmptyImageBox = 1;
	else suppressEmptyImageBox = 0;
	if (dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressUnassignedBoxCheck.value == true) suppressUnassignedBox = 1;
	else suppressUnassignedBox = 0;
	if (dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressLineBoxCheck.value == true) suppressLineBox = 1;
	else suppressLineBox = 0;
	if (dlgw.leftGroup.tabPanel.tab4.excludes2Group.suppressNonPrintingBoxCheck.value == true) suppressNonPrintingBox = 1;
	else suppressNonPrintingBox = 0;

	//----------- tab5 fields: Tables
	TABLE_BORDER_COLLAPSE = dlgw.leftGroup.tabPanel.tab5.tableOptionsGroup.TABLE_BORDER_COLLAPSEDropdown.selection.index;
	TABLE_CELLS_WIDTH_PRESERVE = dlgw.leftGroup.tabPanel.tab5.cellResizeGroup.TABLE_CELLS_WIDTH_PRESERVEField.text;

	//----------- tab6 fields: View
	cssName = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.cssNameGroup.cssNameField.text;
	if ( (cssName != "") && (endsWith(cssName.toLowerCase(),".css") == false)) cssName += ".css";
	if (dlgw.leftGroup.tabPanel.tab6.top.leftGroup.CSSclassNamesASCIICheck.value == true) CSSclassNamesASCII = 1;
	else CSSclassNamesASCII = 0;

	//**** magnifying: copy original JPEG exported from InDesign
	magnifyingGlass = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.viewModeGroup.viewModeDropdown.selection.index;

	suppressExportMouseOvers = "";
	if (dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressAllMouseOversCheck.value == true) suppressExportMouseOvers += "1";
	if (dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressTextMouseOversCheck.value == true) suppressExportMouseOvers += "2";
	if (dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressImageMouseOversCheck.value == true) suppressExportMouseOvers += "4";
	if (dlgw.leftGroup.tabPanel.tab6.top.rightGroup.suppressEmptyMouseOversCheck.value == true) suppressExportMouseOvers += "8";
	try {
		fontsizeFactor = parseFloat(dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeFactorGroup.fontsizeFactorField.text);
	} catch(e) {
		fontsizeFactor = 1.0;
	}
	/****** in test phase
	fontsizeUnits = dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeUnitsGroup.fontsizeUnitsDropdown.selection.index;
	try {
		fontsizeBase = parseFloat(dlgw.leftGroup.tabPanel.tab6.top.leftGroup.fontsizeBaseGroup.fontsizeBaseField.text);
	} catch(e) {
		fontsizeBase = 1.0;
	}
	******/

	websiteParams = dlgw.leftGroup.tabPanel.tab6.bottom.websiteParamsGroup.websiteParamsField.text;

	//----------- tab7 fields: Paths
	if (dlgw.leftGroup.tabPanel.pathInfosTab.transformerPathGroup.startTransformerCheck.value == true) doStartBatchXSLT = true;
	else doStartBatchXSLT = false;
	if (dlgw.leftGroup.tabPanel.pathInfosTab.communicationPathGroup.controlTransformerCheck.value == true) doControlBatchXSLT = true;
	else doControlBatchXSLT = false;


	/* -----currently disabled
	mainExportFolderInTransformer = dlgw.leftGroup.tabPanel.pathInfosTab.srcExportPathGroupTransf.path.text;
	mainExportFolderOutTransformer = dlgw.leftGroup.tabPanel.pathInfosTab.xmlExportPathGroupTransf.path.text;
	*/


	//----------- bottom: actions after done
	if (dlgw.leftGroup.actionsGroup.closeDocAfterDoneCheck.value == true) closeDocumentAfterDone = true;
	else closeDocumentAfterDone = false;
	if (dlgw.leftGroup.actionsGroup.showExportFolderCheck.value == true) showExportFolder = true;
	else showExportFolder = false;
	if (dlgw.leftGroup.actionsGroup.showMessageCheck.value == true) showMessage = true;
	else showMessage = false;


	// should current settings be saved?
	if ((go == 3) || (dlgw.leftGroup.actionsGroup.saveSettingsCheck.value == true)) {
		saveSettings = true;
		saveSettingsToFile();
	}
	else saveSettings = false;

	//alert("go: " + go);

	try { dlgw.close();} catch(e) {}

	// after show main dialog: run hook
	runHooks(scriptName,"afterShowMainDialog", {"go":go});

	return(go);
}


function checkDialogFields() {
	mainDialogErrorMessage = "";
	mainDialogErrorCode = 0;
	if ( issueYear.length != 0 ) {
		if ( issueYear.length != 4)  {
			//mainDialogErrorMessage = "'Year' field must contain 4 digits (YYYY) like 2008";
			mainDialogErrorMessage = loc(20);
			mainDialogErrorCode = 1;
		}
	}
	if ( issueDate.length != 0 ) {
		if ( issueDate.length != 8 ) {
			//mainDialogErrorMessage = "'Date' field must contain 8 digits (YYYYMMDD) like 20080324";
			mainDialogErrorMessage = loc(21);
			mainDialogErrorCode = 2;
		}
	}

	build_export_paths("","checkDialogFields",suppress_exportSubPath);	// enhance export paths with subpath

	return(mainDialogErrorCode);
}


function settingsFilePathWritable() {	// is settings path writable? otherwise return alternate settings path

	var canwrite = canWriteInFolder(settingsFilePath, true);
	//alert("settings path: " + settingsFilePath + "\ncanwrite: " + canwrite + "\nerror.text: " + error.text);
	if (canwrite == true) {	// we can write
		return(settingsFilePath);
	}
	else {
		// check if we can write into InDesign's scripting folder
		var scriptspath = getScriptsPath() + folderSeparator + settingsFilePathAlt;
		canwrite = canWriteInFolder(scriptspath, false);
		//alert("settings path: " + scriptspath + "\ncanwrite: " + canwrite + "\nerror.text: " + error.text);
		if (canwrite == true) {	// we can write
			var checkFolder = new Folder(scriptspath);
			//alert("writable: " + checkFolder.fsName);
			return(checkFolder.fsName);
		}
	}
	return(settingsFilePath);	// uuuups
}

function canWriteInFolder(path, createIfNotExists) {
	error.text = "";
	error.lastcode = 0;
	var mypath = path;
	if ( endsWith(mypath,folderSeparator) == false) mypath += folderSeparator;
	var fldr = new Folder(mypath);	// make sure the path exists
	if (!fldr.exists) {
		if (createIfNotExists == false) return(false);
		var fldrcreated = fldr.create();
		if (fldrcreated == false) {
			error.lastcode = error.FOLDER_CREATE;
			error.text = fldr.error;
			//alert("canWriteInFolder()\ncould not create path: " + path);
			return(false);	// we can NOT create/write
		}
	}
	// the folder exists or was created
	var checkFile = new File(mypath + folderSeparator + "writecheck__.tst");
	var created = checkFile.open("w");
	if (created == true) {	// we can write
		checkFile.close();
		checkFile.remove();
		//alert("canWriteInFolder()\nwritable path: " + path);
		return(true);
	}
	error.lastcode = error.FILE_CREATE;
	error.text = checkFile.error;
	//alert("canWriteInFolder()\nNON writable path: " + path);
	return(false);
}

function copySettingsFiles() { // copy all settings files from InDesign's scripting folder into the writable communications folder if not already there
	var srcfldr = new Folder(getScriptsPath() + folderSeparator + settingsFilePathAlt);
	var targfldr = new Folder(settingsFilePath);	// make sure the settings path exists
	if (targfldr.exists == false) return(-1);
	var created = targfldr.create();	// make sure the target settings path exists
	//alert("copySettingsFiles:" + "\nsrcfldr: " + srcfldr + "\ntargfldr: " + targfldr + "\ncreated: " + created);
	var settingsFiles = srcfldr.getFiles("*");

	for (var i = 0; i < settingsFiles.length; i++) {
		if (settingsFiles[i].fsName.indexOf(".DS_Store") >= 0) continue;
		if (endsWith(settingsFiles[i].fsName,".ini") == true) continue;
		var srcSettingsFilePathName = settingsFiles[i];
		var targSettingsFilePathName = targfldr.fsName + folderSeparator + settingsFiles[i].name;

		var tf = new File(targSettingsFilePathName);	// check if target already exists
		if (tf.exists == true) continue;	// comment this line, if the settings always should be overwritten by application settings

		var sf = new File(srcSettingsFilePathName);
		var success = sf.copy(targSettingsFilePathName);
		/*
		alert("copying settingsFileName: " + settingsFiles[i].name
			+ "\nfrom\nsrcSettingsFilePathName: " + srcSettingsFilePathName
			+ "\nto\ntargSettingsFilePathName: " + targSettingsFilePathName
			+ "\n\nsuccess: " + success
			);
		*/
	}
	return(0);
}

function saveSettingsToFile() {
	if (dontAutoSaveSettingsFile == true) return;
	//if (settingsloaded != 0) return; // don't save back if it could not be loaded
	// write an init file if settingsFilePath != settingsFilePathDefault
	if ((settingsFilePath != settingsFilePathDefault) && (callerScript == "")) saveInitToFile();

	// write the current settings to disk
	var thesettingspath = settingsFilePath;
	if (endsWith(thesettingspath,folderSeparator) == false) thesettingspath += folderSeparator;
	thesettingspath += settingsFileName;
	var settingsFile = new File(thesettingspath);
	settingsFile.encoding = "UTF-8";
	//alert("saving settings to: " + settingsFile.fsName);
	try { settingsFile.remove(); } catch(e) {}
	//alert(settingsFile.fsName);
	var err = settingsFile.open("w");

	err = settingsFile.write("settingsFilePath=" + getURIName(settingsFilePath) + "\n");

	if (BXSLT4InDesignBasePath != BXSLT4InDesignBasePathDefault) err = settingsFile.write("BXSLT4InDesignBasePath=" + BXSLT4InDesignBasePath + "\n");
	err = settingsFile.write("dontAutoSaveSettingsFile=" + dontAutoSaveSettingsFile + "\n");
	err = settingsFile.write("silentMode=" + silentMode + "\n");

	err = settingsFile.write("BXSLT4InDesignApplRunsOnWindows=" + BXSLT4InDesignApplRunsOnWindows + "\n");
	err = settingsFile.write("BXSLT4InDesignApplRunsOnOSX=" + BXSLT4InDesignApplRunsOnOSX + "\n");
	err = settingsFile.write("BXSLT4InDesignApplRunsOnUnix=" + BXSLT4InDesignApplRunsOnUnix + "\n");
	err = settingsFile.write("servernameOnWindows=" + servernameOnWindows + "\n");
	err = settingsFile.write("servernameOnOSX=" + servernameOnOSX + "\n");
	err = settingsFile.write("servernameOnUnix=" + servernameOnUnix + "\n");

	err = settingsFile.write("BXSLT4InDesignCommDir=" + getURIName(BXSLT4InDesignCommDir) + "\n");
	err = settingsFile.write("doStartBatchXSLT=" + doStartBatchXSLT + "\n");
	err = settingsFile.write("doControlBatchXSLT=" + doControlBatchXSLT + "\n");
	err = settingsFile.write("hooksBasePath=" + getURIName(hooksBasePath) + "\n");
	err = settingsFile.write("mainExportFolder=" + getURIName(mainExportFolder) + "\n");

	err = settingsFile.write("mainExportFolderInTransformer=" + mainExportFolderInTransformer + "\n");
	err = settingsFile.write("mainExportFolderOutTransformer=" + mainExportFolderOutTransformer + "\n");

	err = settingsFile.write("companyName=" + companyName + "\n");
	err = settingsFile.write("objectName=" + objectName + "\n");
	err = settingsFile.write("objectShortcut=" + objectShortcut + "\n");
	err = settingsFile.write("issueYear=" + issueYear + "\n");
	err = settingsFile.write("issueDate=" + issueDate + "\n");
	err = settingsFile.write("fileDateFormat=" + fileDateFormat + "\n");
	err = settingsFile.write("cssName=" + cssName + "\n");
	err = settingsFile.write("fontsizeFactor=" + fontsizeFactor + "\n");
	err = settingsFile.write("fontsizeUnits=" + fontsizeUnits + "\n");
	err = settingsFile.write("fontsizeBase=" + fontsizeBase + "\n");
	err = settingsFile.write("CSSclassNamesASCII=" + CSSclassNamesASCII + "\n");
	err = settingsFile.write("magnifyingGlass=" + magnifyingGlass + "\n");

	err = settingsFile.write("outputMode=" + outputMode + "\n");
	err = settingsFile.write("outputModeCustomXSL=" + outputModeCustomXSL + "\n");
	err = settingsFile.write("pageTurnMode=" + pageTurnMode + "\n");
	err = settingsFile.write("exportImages=" + exportImages + "\n");
	err = settingsFile.write("exportPageJPEGs=" + exportPageJPEGs + "\n");
	err = settingsFile.write("exportPagePDFs=" + exportPagePDFs + "\n");
	err = settingsFile.write("exportPagePDFusePresetName=" + exportPagePDFusePresetName + "\n");
	err = settingsFile.write("exportDocumentPDF=" + exportDocumentPDF + "\n");
	err = settingsFile.write("exportDocumentPDFspreads=" + exportDocumentPDFspreads + "\n");
	err = settingsFile.write("exportDocumentPDFusePresetName=" + exportDocumentPDFusePresetName + "\n");

	err = settingsFile.write("excludeHiddenLayers=" + excludeHiddenLayers + "\n");
	err = settingsFile.write("excludeLayers=" + excludeLayers + "\n");
	err = settingsFile.write("suppressAllEmptyBox=" + suppressAllEmptyBox + "\n");
	err = settingsFile.write("suppressEmptyTextBox=" + suppressEmptyTextBox + "\n");
	err = settingsFile.write("suppressEmptyImageBox=" + suppressEmptyImageBox + "\n");
	err = settingsFile.write("suppressUnassignedBox=" + suppressUnassignedBox + "\n");
	err = settingsFile.write("suppressLineBox=" + suppressLineBox + "\n");
	err = settingsFile.write("suppressNonPrintingBox=" + suppressNonPrintingBox + "\n");

	err = settingsFile.write("suppressExportMouseOvers=" + suppressExportMouseOvers + "\n");

	err = settingsFile.write("pageframingBoxNoExport=" + pageframingBoxNoExport + "\n");
	err = settingsFile.write("pageframingBoxSizeThreshold=" + pageframingBoxSizeThreshold + "\n");
	err = settingsFile.write("pageframingBoxesNoCatch=" + pageframingBoxesNoCatch + "\n");

	err = settingsFile.write("exportPROFeatures=" + exportPROFeatures + "\n");
	err = settingsFile.write("includeXMLtags=" + includeXMLtags + "\n");

	err = settingsFile.write("imageAddTargetPrefix=" + imageAddTargetPrefix + "\n");
	err = settingsFile.write("imageOverwriteTargetIfExists=" + imageOverwriteTargetIfExists + "\n");
	err = settingsFile.write("imageInputParam=" + imageInputParam + "\n");
	err = settingsFile.write("imageDPI=" + imageDPI + "\n");
	err = settingsFile.write("imageQUALITY=" + imageQUALITY + "\n");
	err = settingsFile.write("imageMissingCreateFromPagePreview=" + imageMissingCreateFromPagePreview + "\n");

	err = settingsFile.write("extractImageMetaData=" + extractImageMetaData + "\n");
	err = settingsFile.write("fullImageMetaData=" + fullImageMetaData + "\n");

	err = settingsFile.write("imageParameters=" + imageParameters + "\n");
	err = settingsFile.write("imageOutputFinishParam=" + imageOutputFinishParam + "\n");
	err = settingsFile.write("copyOriginalImageFiles=" + copyOriginalImageFiles + "\n");
	err = settingsFile.write("imageDoROTATE=" + imageDoROTATE + "\n");
	err = settingsFile.write("imageEXCLUDE=" + imageEXCLUDE + "\n");

	err = settingsFile.write("pageImageWidth=" + pageImageWidth + "\n");
	err = settingsFile.write("pageImageDPI=" + pageImageDPI + "\n");
	err = settingsFile.write("pageImageQuality=" + pageImageQuality + "\n");
	err = settingsFile.write("pageJPEGParams=" + pageJPEGParams + "\n");
	err = settingsFile.write("pageJPEGfinish=" + pageJPEGfinish + "\n");

	err = settingsFile.write("splittingStoriesOption=" + splittingStoriesOption + "\n");

	err = settingsFile.write("catchRadius=" + catchRadius + "\n");
	err = settingsFile.write("allBoxesNoCatch=" + allBoxesNoCatch + "\n");
	err = settingsFile.write("unassignedBoxesNoCatch=" + unassignedBoxesNoCatch + "\n");
	err = settingsFile.write("emptyBoxesNoCatch=" + emptyBoxesNoCatch + "\n");
	err = settingsFile.write("textBoxesNoCatch=" + textBoxesNoCatch + "\n");
	err = settingsFile.write("chainedBoxesNoCatch=" + chainedBoxesNoCatch + "\n");
	err = settingsFile.write("imageBoxesNoCatch=" + imageBoxesNoCatch + "\n");


	err = settingsFile.write("TABLE_BORDER_COLLAPSE=" + TABLE_BORDER_COLLAPSE + "\n");
	err = settingsFile.write("TABLE_CELLS_WIDTH_PRESERVE=" + TABLE_CELLS_WIDTH_PRESERVE + "\n");

	err = settingsFile.write("websiteParams=" + websiteParams + "\n");

	err = settingsFile.write("runBeforeJobApp=" + runBeforeJobApp + "\n");
	err = settingsFile.write("runAfterJobApp=" + runAfterJobApp + "\n");
	err = settingsFile.write("runBeforeTransformApp=" + runBeforeTransformApp + "\n");
	err = settingsFile.write("runAfterTransformApp=" + runAfterTransformApp + "\n");

	err = settingsFile.write("closeDocumentAfterDone=" + closeDocumentAfterDone + "\n");
	err = settingsFile.write("showExportFolder=" + showExportFolder + "\n");
	err = settingsFile.write("showMessage=" + showMessage + "\n");
	err = settingsFile.write("saveSettings=" + saveSettings + "\n");

	err = settingsFile.write("\n------ End of settings \n");	// ending empty line!!
	err = settingsFile.close();
}


function loadSettingsFromFile() {
	// load the current settings from disk
	var settingsFile = new File(settingsFilePath + folderSeparator + settingsFileName);
	//alert("loadSettingsFromFile: " + settingsFile.fsName + "\nexists: " + settingsFile.exists);
	if (settingsFile.exists == false) return(-1);
	settingsFile.encoding = "UTF-8";
	var err = settingsFile.open("r");
	//alert("settings file:" + settingsFile.fsName);
	var line = "";
	while (!settingsFile.eof) {
		line = settingsFile.readln();
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			var args_arr = new Array(key,arg);
			switch(args_arr[0]) {
				case "settingsFilePath": if (args_arr[1] != "") settingsFilePath = args_arr[1]; break;
				// DO NOT SET THIS! USER MIGHT CHANGE THE LOCATION TO PROGRAM FROM APPLICATIONS TO USER HOME AND VICE VERSA
				// THE SCRIPT ALWAYS MUST SEARCH FOR THE CURRENT TRANSFORMER APPLICATION!
				//	case "BXSLT4InDesignBasePath": BXSLT4InDesignBasePath = args_arr[1]; break;
				case "dontAutoSaveSettingsFile": if (args_arr[1] == "true") dontAutoSaveSettingsFile = true; else dontAutoSaveSettingsFile = false; break;
				case "silentMode": silentMode = parseInt(args_arr[1]); break;

				case "BXSLT4InDesignApplRunsOnWindows": BXSLT4InDesignApplRunsOnWindows = args_arr[1]; break;
				case "BXSLT4InDesignApplRunsOnOSX": BXSLT4InDesignApplRunsOnOSX = args_arr[1]; break;
				case "BXSLT4InDesignApplRunsOnUnix": BXSLT4InDesignApplRunsOnUnix = args_arr[1]; break;
				case "servernameOnWindows": servernameOnWindows = args_arr[1]; break;
				case "servernameOnOSX": servernameOnOSX = args_arr[1]; break;
				case "servernameOnUnix": servernameOnUnix = args_arr[1]; break;

				case "BXSLT4InDesignCommDir": if (args_arr[1] != "") BXSLT4InDesignCommDir = args_arr[1]; break;
				case "doStartBatchXSLT": if (args_arr[1] == "true") doStartBatchXSLT = true; else doStartBatchXSLT = false; break;
				case "doControlBatchXSLT": if (args_arr[1] == "true") doControlBatchXSLT = true; else doControlBatchXSLT = false; break;

				case "hooksBasePath": hooksBasePath = args_arr[1]; ;break;

				case "mainExportFolder":
					var mymainExportFolder = getFsName(args_arr[1]);
					var canwrite = false;
					// check if path exists and is writable
					var mypath = new Folder(mymainExportFolder);
					canwrite = canWriteInFolder(mymainExportFolder, true);	// try to create it
					//alert("2:export folder: " + mypath.path + "\nexists: " + mypath.exists + "\ncanwrite: " + canwrite);
					if ((mypath.exists == true) && (canwrite == true)) mainExportFolder = mymainExportFolder
					break;

				case "mainExportFolderInTransformer": mainExportFolderInTransformer = args_arr[1]; break;
				case "mainExportFolderOutTransformer": mainExportFolderOutTransformer = args_arr[1]; break;

				case "companyName": companyName = args_arr[1]; break;
				case "objectName": objectName = args_arr[1]; break;
				case "objectShortcut": objectShortcut = args_arr[1]; break;
				case "issueYear": issueYear = args_arr[1]; break;
				case "issueDate": issueDate = args_arr[1]; break;
				case "fileDateFormat": fileDateFormat = args_arr[1]; break;
				case "cssName": cssName = args_arr[1]; break;
				case "fontsizeFactor": try { fontsizeFactor = parseFloat(args_arr[1]); } catch(e){ fontsizeFactor = 1.0; } break;
				case "fontsizeUnits": 
					try {
						if (isNaN(args_arr[1])) fontsizeUnits = 0;
						else fontsizeUnits = parseInt(args_arr[1]);
					} catch(e){ fontsizeUnits = 0; }
					break;
				case "fontsizeBase": try { fontsizeBase = parseFloat(args_arr[1]); } catch(e){ fontsizeBase = 1.0; } break;

				case "CSSclassNamesASCII": CSSclassNamesASCII = parseInt(args_arr[1]); break;
				case "magnifyingGlass": magnifyingGlass = parseInt(args_arr[1]); break;
				case "outputMode": outputMode = parseInt(args_arr[1]); break;
				case "outputModeCustomXSL": outputModeCustomXSL = args_arr[1]; break;
				case "turnPageMode": turnPageMode = args_arr[1]; break;
				case "exportImages":
					if (isNaN(args_arr[1][0])) {	// compatibility for true/false
						if (args_arr[1] == "true") exportImages = 1;
						else exportImages = 0;
					}
					else {	// 0 = dont'export images, 1 = JPEG, 2 = GIF, 3 = PNG, 4 = TIFF
						exportImages = parseInt(args_arr[1]);
					}
					break;
				case "exportPageJPEGs": if (args_arr[1] == "true") exportPageJPEGs = true; else exportPageJPEGs = false; break;
				case "exportPagePDFs": if (args_arr[1] == "true") exportPagePDFs = true; else exportPagePDFs = false; break;
				case "exportPagePDFusePresetName": {
					exportPagePDFusePresetName = args_arr[1];
					if (exportPagePDFusePresetName != "") {
						exportPagePDFusePresetIndex = getPDFpresetIndexByName(exportPagePDFusePresetName);
						if (exportPagePDFusePresetIndex < 0) exportPagePDFusePresetName = "";
					}
					else {
						exportPagePDFusePresetIndex = 0;
						exportPagePDFusePresetName = exportPDFpresetsNames[exportPagePDFusePresetIndex];
					}
					break;
					}
				case "exportDocumentPDF": if (args_arr[1] == "true") exportDocumentPDF = true; else exportDocumentPDF = false; break;
				case "exportDocumentPDFspreads": if (args_arr[1] == "true") exportDocumentPDFspreads = true; else exportDocumentPDFspreads = false; break;
				case "exportDocumentPDFusePresetName": {
					exportDocumentPDFusePresetName = args_arr[1];
					if (exportDocumentPDFusePresetName != "") {
						exportDocumentPDFusePresetIndex = getPDFpresetIndexByName(exportDocumentPDFusePresetName);
						if (exportDocumentPDFusePresetIndex < 0) exportDocumentPDFusePresetName = "";
					}
					else {
						exportDocumentPDFusePresetIndex = 0;
						exportDocumentPDFusePresetName = exportPDFpresetsNames[exportDocumentPDFusePresetIndex];
					}
					break;
					}

				case "excludeHiddenLayers": excludeHiddenLayers = parseInt(args_arr[1]); break;
				case "excludeLayers": excludeLayers = args_arr[1]; break;
				case "suppressAllEmptyBox": suppressAllEmptyBox = parseInt(args_arr[1]); break;
				case "suppressEmptyTextBox": suppressEmptyTextBox = parseInt(args_arr[1]); break;
				case "suppressEmptyImageBox": suppressEmptyImageBox = parseInt(args_arr[1]); break;
				case "suppressUnassignedBox": suppressUnassignedBox = parseInt(args_arr[1]); break;
				case "suppressLineBox": suppressLineBox = parseInt(args_arr[1]); break;
				case "suppressNonPrintingBox": suppressNonPrintingBox = parseInt(args_arr[1]); break;

				case "suppressExportMouseOvers": suppressExportMouseOvers = args_arr[1]; break;

				case "pageframingBoxNoExport": pageframingBoxNoExport = parseInt(args_arr[1]); break;
				case "pageframingBoxSizeThreshold": pageframingBoxSizeThreshold = parseFloat(args_arr[1]); break;
				case "pageframingBoxesNoCatch": pageframingBoxesNoCatch = parseInt(args_arr[1]); break;

				case "exportPROFeatures": if (args_arr[1] == "true") exportPROFeatures = true; else exportPROFeatures = false; break;
				case "includeXMLtags": 
					if (enableXMLtagsMenu) {
						includeXMLtags = parseInt(args_arr[1]); break;
					}
				case "imageScale": imageParameters = args_arr[1]; break;	// backward compatibility
				case "imageInputParam": imageInputParam = args_arr[1]; break;
				case "imageDPI": imageDPI = args_arr[1]; break;
				case "imageQUALITY": imageQUALITY = args_arr[1]; break;
				// when IMageMagick is installed: case "imageMissingCreateFromPagePreview": imageMissingCreateFromPagePreview = parseInt(args_arr[1]); break;

				case "extractImageMetaData": extractImageMetaData = parseInt(args_arr[1]); break;
				case "fullImageMetaData": fullImageMetaData = parseInt(args_arr[1]); break;

				case "imageAddTargetPrefix": imageAddTargetPrefix = parseInt(args_arr[1]); break;
				case "imageOverwriteTargetIfExists": imageOverwriteTargetIfExists = parseInt(args_arr[1]); break;
				case "imageParameters": imageParameters = args_arr[1]; break;
				case "imageOutputFinishParam": imageOutputFinishParam = args_arr[1]; break;
				case "copyOriginalImageFiles":
					if ((args_arr[1] == "true") || (args_arr[1] == "false")) {	// compatibiliy for older settings
						if (args_arr[1] == "true") copyOriginalImageFiles = 1;
						else copyOriginalImageFiles = 0;
					}
					else copyOriginalImageFiles = parseInt(args_arr[1]);
					break;
				case "imageDoROTATE": imageDoROTATE = parseInt(args_arr[1]); break;
				case "imageEXCLUDE": imageEXCLUDE = args_arr[1]; break;

				case "pageImageWidth": pageImageWidth = args_arr[1]; break;
				case "pageImageDPI": pageImageDPI = args_arr[1]; break;
				case "pageImageQuality": pageImageQuality = args_arr[1]; break;
				case "pageJPEGParams": pageJPEGParams = args_arr[1]; break;
				case "pageJPEGfinish": pageJPEGfinish = args_arr[1]; break;

				case "splittingStoriesOption": splittingStoriesOption = parseInt(args_arr[1]); break;

				case "catchRadius": catchRadius = parseInt(args_arr[1]); break;
				case "allBoxesNoCatch": allBoxesNoCatch = parseInt(args_arr[1]); break;
				case "unassignedBoxesNoCatch": if (args_arr[1] == "true") unassignedBoxesNoCatch = true; else unassignedBoxesNoCatch = false; break;
				case "emptyBoxesNoCatch": if (args_arr[1] == "true") emptyBoxesNoCatch = true; else emptyBoxesNoCatch = false; break;
				case "textBoxesNoCatch": if (args_arr[1] == "true") textBoxesNoCatch = true; else textBoxesNoCatch = false; break;
				case "chainedBoxesNoCatch": if (args_arr[1] == "true") chainedBoxesNoCatch = true; else chainedBoxesNoCatch = false; break;
				case "imageBoxesNoCatch": if (args_arr[1] == "true") imageBoxesNoCatch = true; else imageBoxesNoCatch = false; break;

				case "TABLE_BORDER_COLLAPSE": TABLE_BORDER_COLLAPSE = args_arr[1]; break;
				case "TABLE_CELLS_WIDTH_PRESERVE": TABLE_CELLS_WIDTH_PRESERVE = args_arr[1]; break;

				case "websiteParams": websiteParams = args_arr[1]; break;

				case "runBeforeJobApp": runBeforeJobApp = args_arr[1]; break;
				case "runAfterJobApp": runAfterJobApp = args_arr[1]; break;
				case "runBeforeTransformApp": runBeforeTransformApp = args_arr[1]; break;
				case "runAfterTransformApp": runAfterTransformApp = args_arr[1]; break;

				case "closeDocumentAfterDone": if (args_arr[1] == "true") closeDocumentAfterDone = true; else closeDocumentAfterDone = false; break;
				case "showExportFolder": if (args_arr[1] == "true") showExportFolder = true; else showExportFolder = false; break;
				case "showMessage": if (args_arr[1] == "true") showMessage = true; else showMessage = false; break;
				case "saveSettings": if (args_arr[1] == "true") saveSettings = true; else saveSettings = false; break;

				case "allowRevertDocument": if (args_arr[1] == "true") allowRevertDocument = true; else allowRevertDocument = false; break;
				case "BatchXSLTtriggerMode": BatchXSLTtriggerMode = parseInt(args_arr[1]); break;
			}
		}
	}
	err = settingsFile.close();
	build_mainExportFolders();
	return(0);
}

function saveInitToFile() {
	// check if we have to write an init file
	var sf = new Folder(settingsFilePath);
	var sfdef = new Folder(settingsFilePathDefault);
	//alert(sf.fsName + "\n" + sfdef.fsName);
	if (sf.fsName.indexOf(sfdef.fsName) >= 0) return;
	// write the INIT file
	var initPathName = "~/" + initFileName;
	var initFile = new File(initPathName);
	try { initFile.remove(); } catch(e) {}
	//alert(settingsFile.fsName);
	var err = initFile.open("w");
	var f = new Folder(settingsFilePath);
	//alert("saveInitToFile()\nsettingsFilePath=" + f.fsName);
	err = initFile.write("settingsFilePath=" + f.fsName + "\n");
	err = initFile.write("settingsFileName=" + settingsFileName + "\n");
	err = initFile.close();
}
function removeInitFile() {
	initFileNameLoaded = "";

	// delete the INIT file
	var initPathName = "~/" + initFileName;
	var initFile = new File(initPathName);
	try { initFile.remove(); } catch(e) {}
}

function loadInitFromFile() {
	// try to load from user home path
	var initPath = "~/" + initFileName;
	var initFile = new File(initPath);
	if (initFile.exists == false) {
		// try to load the init file from settings path
		initPath = settingsFilePath;
		if (endsWith(initPath,folderSeparator) == false) initPath += folderSeparator;
		initPath += initFileName;
		initFile = new File(initPath);
		if (initFile.exists == false) {
			// try to load from application scripts path
			initPath = getScriptsPath() + folderSeparator + settingsFilePathAlt;
			if ((endsWith(initPath,folderSeparator) == false) && (startsWith(settingsFilePathAlt,folderSeparator) == false)) initPath += folderSeparator;
			initPath += initFileName;
			initFile = new File(initPath);
			if (initFile.exists == false) return(-1);
		}
	}
	//alert("loadInitFromFile:\n" + initFile.fsName);
	initFileNameLoaded = initFile.fsName;        // store path/name of init file
	var err = initFile.open("r");

	var line = "";
	while (!initFile.eof) {
		line = initFile.readln();
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			var args_arr = new Array(key,arg);
			//alert(args_arr);
			switch(args_arr[0]) {
				case "settingsFilePath": settingsFilePath = getFsName(args_arr[1]); break;
				case "settingsFileName": settingsFileName = args_arr[1]; break;
			}
		}
	}
	err = initFile.close();
	return(0);
}


function factorySettings() {
	// remove an evtl. existing init file
	removeInitFile();

	// set defaults
	dontAutoSaveSettingsFile= false;
	silentMode = 0;
	BXSLT4InDesignApplRunsOnWindows = "";
	BXSLT4InDesignApplRunsOnOSX = "";
	BXSLT4InDesignApplRunsOnUnix = "";
	servernameOnWindows = "\\\\servername\\";
	servernameOnOSX = "/Volumes/";
	servernameOnUnix = "//servername/";
	doStartBatchXSLT = true;
	doControlBatchXSLT = true;

	settingsFileName = settingsFileNameDefault;
	settingsFilePath = settingsFilePathDefault;
	BXSLT4InDesignCommDir = BXSLT4InDesignCommDirDefault;
	BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault = "~/" + applicationProgramName + packageExtension + folderSeparator + tranformerName + folderSeparator;

	mainExportFolder = "~/Export";
	build_mainExportFolders();

	companyName = "";
	objectName = "";
	objectShortcut = "";
	issueYear = "";
	issueDate = "";
	fileDateFormat = "";
	cssName = "";
	fontsizeFactor = 1.0;
	fontsizeUnits = 0;
	fontsizeBase = 9.0;
	CSSclassNamesASCII = 0;
	magnifyingGlass = 0;
	outputMode = 0;
	outputModeCustomXSL = "";
	pageTurnMode = 'turn';
	exportImages = 1;
	exportPageJPEGs = true;
	exportPagePDFs = true;
	exportPagePDFusePresetName = "";
	exportDocumentPDF = false;
	exportDocumentPDFspreads = false;
	exportDocumentPDFusePresetName = "";

	excludeHiddenLayers = 1;
	excludeLayers = "noexp";
	suppressAllEmptyBox = 0;
	suppressEmptyTextBox = 0;
	suppressEmptyImageBox = 0;
	suppressUnassignedBox = 0;
	suppressLineBox = 1;
	suppressNonPrintingBox = 1;

	pageframingBoxNoExport = 12;
	pageframingBoxSizeThreshold = 1.0;
	pageframingBoxesNoCatch = 1;

	suppressExportMouseOvers = "8";

	exportPROFeatures = false;
	includeXMLtags = 0;

	imageAddTargetPrefix = 1;
	imageOverwriteTargetIfExists = 1;
	imageInputParam = "";
	imageInputDPI = "150";
	imageDPI = "150";
	imageQUALITY = "90";
	imageParameters = "1.0";
	imageOutputFinishParam = "-strip";
	copyOriginalImageFiles = 0;
	imageDoROTATE = 0;
	imageEXCLUDE = "excl_;555";
	imageMissingCreateFromPagePreview = 2;

	extractImageMetaData = 0;
	fullImageMetaData = 0;

	pageImageWidth = "500//800";
	pageImageDPI = "300";
	pageImageQuality = "90";
	pageJPEGParams = "";
	pageJPEGfinish = "";

	splittingStoriesOption = 0;

	catchRadius = 0;
	allBoxesNoCatch = 0;
	unassignedBoxesNoCatch = true;
	emptyBoxesNoCatch = false;
	textBoxesNoCatch = false;
	chainedBoxesNoCatch = true;
	chainedFirstBoxesNoCatch = true;
	chainedFollowingBoxesNoCatch = true;
	imageBoxesNoCatch = false;

	TABLE_BORDER_COLLAPSE = "1";
	TABLE_CELLS_WIDTH_PRESERVE = "1.3";

	websiteParams = "logoName=*#*logoURL=*#*logoTitle=*#*logoURLtarget=";

	runBeforeJobApp = "";
	runAfterJobApp = "";
	runBeforeTransformApp = "";
	runAfterTransformApp = "";

	closeDocumentAfterDone = false;
	showExportFolder = true;
	showMessage = true;
	saveSettings = true;

	allowRevertDocument = true;
	BatchXSLTtriggerMode = 0;

	// init some vars
	var currentDocument = "";
	var currentDocumentName = "";
	var currentDocumentPath = "";
	try { currentDocument = app.activeDocument; } catch(e){}
	try { currentDocumentName = currentDocument.name; } catch(e){}
	try { currentDocumentPath = currentDocument.filePath.toString(); } catch(e){}
	retrieve_issue_year_date(currentDocumentName,currentDocumentPath);
	objectShortcut = retrieve_objectShortcut(currentDocumentName);

	build_export_paths("","factorySettings");
	
}


function loadCommPrefsFromFile() {
	// load the settings to communicate with tranformer from disk
	var commPrefsFile = new File(BXSLT4InDesignCommDir + folderSeparator + "comm" + folderSeparator + "comm.prefs");
	//alert(commPrefsFile);
	if (commPrefsFile.exists == false) return;
	var err = commPrefsFile.open("r");

	var line = "";
	while (!commPrefsFile.eof) {
		line = commPrefsFile.readln();
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			var args_arr = new Array(key,arg);
			//alert(args_arr);
			switch(args_arr[0]) {
				case "BatchXSLTMajorVersion": BatchXSLTMajorVersion = args_arr[1]; break;
				case "BatchXSLTMinorVersion": BatchXSLTMinorVersion = args_arr[1]; break;
				case "BatchXSLTRequiredXslVersionName": BatchXSLTRequiredXslVersionName = args_arr[1]; break;
				case "BatchXSLTAppDir": BXSLT4InDesignBasePathFromPrefs = args_arr[1]; break;
				case "BatchXSLTCommDir": BXSLT4InDesignCommDir = args_arr[1]; break;
			}
		}
	}
	err = commPrefsFile.close();
}


function get_xsl_files() {
	// we first want the default XSLs: 'flipbook.xsl', /* 'xml2htmlEpaper.xsl' obsolete */, 'xml2htmlList.xsl', 'xml2htmlSimpleList.xsl' and 'NO XSL for Tree'
	// even if the XSLCSS master folder cannot be found
	var xslList = new Array();
	xslList[xslList.length] = "flipbook.xsl";			// flip page ebook XML/HTML
// obsolete 	xslList[xslList.length] = "xml2htmlEpaper.xsl";		// epaper page list XML/HTML
	xslList[xslList.length] = "xml2htmlList.xsl";		// Article List XML/HTML
	xslList[xslList.length] = "xml2htmlSimpleList.xsl";	// Article List Simple XML/HTML
	xslList[xslList.length] = "";						// XML tree
// obsolete 	xslList[xslList.length] = "slidebook.xsl";		// slidebook

	var xslcssPath = BXSLT4InDesignBasePath + "XSL" + folderSeparator + "XSLCSS" + folderSeparator;
	//alert("get_xsl_files: " + xslcssPath);
	var f = new Folder(xslcssPath);
	if ((f == null) || (f.exists == false)) {
		//if ( (silentMode >= 0) && !(silentMode & 2) ) alert("Could not find the folder\n'" + xslcssPath + "' of the " + applicationHumanName + " application!\nExport aborted");
		return (xslList);
	}
	var fullXSLlist = f.getFiles("*.xsl");
	// now we add the 'custom' XSLs
	for (var i = 0; i < fullXSLlist.length; i++) {
		if (	(fullXSLlist[i].name == "flipbook.xsl")
			 || (fullXSLlist[i].name == "xml2htmlEpaper.xsl")
			 || (fullXSLlist[i].name == "xml2htmlList.xsl")
			 || (fullXSLlist[i].name == "xml2htmlSimpleList.xsl")
			 || (fullXSLlist[i].name == "slidebook.xsl")
			) {
			continue;
		}
		xslList[xslList.length] = fullXSLlist[i].name;
	}

	return(xslList);
}

function textConditionsVisible() {
	try {
		var conditionText = app.activeDocument.conditions.firstItem();
		while(conditionText != null) {
			//alert(conditionText.name + ": " + conditionText.visible);
			if (conditionText.visible != true) conditionText.visible = true;
			conditionText = app.activeDocument.conditions.nextItem(conditionText);
		}
	}
	catch(e){}
}



function getPDFpresetIndexByName(presetName) {
	if ((exportPDFpresetsNames == null) || (presetName == null) || (presetName == "")) return(-1);
	for (var i = 0; i < exportPDFpresetsNames.length; i++) {
		if (exportPDFpresetsNames[i] == presetName) return(i);
	}
	return(-1);	// not found
}


function build_mainExportFolders() {
	mainExportFolderInConst = mainExportFolder + folderSeparator + "in";
	mainExportFolderIn = mainExportFolderInConst;
	if (jtSubFolder != "") {	// in batch mode add the source main folder
		if ((endsWith(mainExportFolderIn,folderSeparator) == false) && (startsWith(jtSubFolder,folderSeparator) == false)) mainExportFolderIn += folderSeparator;
		mainExportFolderIn += jtSubFolder;
	}
	mainExportFolderOutConst = mainExportFolder + folderSeparator + "out";
	mainExportFolderOut = mainExportFolderOutConst;
	if (jtSubFolder != "") {	// in batch mode add the source main folder
		if ((endsWith(mainExportFolderOut,folderSeparator) == false) && (startsWith(jtSubFolder,folderSeparator) == false)) mainExportFolderOut += folderSeparator;
		mainExportFolderOut += jtSubFolder;
	}
	mainExportFolderDoneConst = mainExportFolder + folderSeparator + "done";
	mainExportFolderDone = mainExportFolderDoneConst;
	if (jtSubFolder != "") {	// in batch mode add the source main folder
		if ((endsWith(mainExportFolderDone,folderSeparator) == false) && (startsWith(jtSubFolder,folderSeparator) == false)) mainExportFolderDone += folderSeparator;
		mainExportFolderDone += jtSubFolder;
	}
}


function build_export_paths(overridden_exportSubPath,caller,the_suppress_exportSubPath) {
	//alert("caller: " + caller + "\noverridden_exportSubPath: " + overridden_exportSubPath);
	// build new export paths
	if ((overridden_exportSubPath != null) && (overridden_exportSubPath != "")) {
		exportSubPath = overridden_exportSubPath;
		//alert("exportSubPath from caller: " + exportSubPath);
	}
	else { // no overrides: create the full export paths
		exportSubPath = "";
		if (companyName != "") {
			exportSubPath = folderSeparator + companyName;
		}
		if (objectName != "") {
			exportSubPath += folderSeparator + objectName;
		}
		if (objectShortcut != "") {
			exportSubPath += folderSeparator + objectShortcut;
		}
		if (issueYear != "") {
			exportSubPath += folderSeparator + issueYear;
			if ( endsWith(exportSubPath,folderSeparator) == false) inpath += folderSeparator;
		}
		if (issueDate != "") {
			exportSubPath += folderSeparator + issueDate;
		}
	}
	if (the_suppress_exportSubPath == true) exportSubPath = "";

	mainExportFolderIn = mainExportFolderInConst;
	mainExportFolderOut = mainExportFolderOutConst;
	mainExportFolderDone = mainExportFolderDoneConst;

	var separator = folderSeparator;
	if ( (endsWith(mainExportFolderIn,folderSeparator) == true) || (startsWith(jtSubFolder,folderSeparator) == true) ) separator = "";
	if (jtSubFolder != "") {
		mainExportFolderIn += separator + jtSubFolder;
		mainExportFolderOut += separator + jtSubFolder;
		mainExportFolderDone += separator + jtSubFolder;
	}
	separator = folderSeparator;
	if ( (endsWith(mainExportFolderIn,folderSeparator) == true) || (startsWith(exportSubPath,folderSeparator) == true) ) separator = "";
	if (exportSubPath != "") {
		mainExportFolderIn += separator + exportSubPath;
		mainExportFolderOut += separator + exportSubPath;
		mainExportFolderDone += separator + exportSubPath;
	}
	/*
	alert("build_export_paths():\ncaller: " + caller + "\n\njtSubFolder: " + jtSubFolder
						+ "\n\nmainExportFolderInConst: " + mainExportFolderInConst
						+ "\n\noverridden_exportSubPath: " + overridden_exportSubPath
						+ "\nthe_suppress_exportSubPath: " + the_suppress_exportSubPath
						+ "\n\njtSubFolder: " + jtSubFolder
						+ "\ncompanyName: " + companyName
						+ "\nobjectName: " + objectName
						+ "\nobjectShortcut: " + objectShortcut
						+ "\nissueYear: " + issueYear
						+ "\nissueDate: " + issueDate
						+ "\n\nexportSubPath: " + exportSubPath
						+ "\n\nmainExportFolderIn: " + mainExportFolderIn
						);
	*/
}


function getScriptsPath() {
	var fullpath = app.activeScript,
		scriptFile = new File(fullpath);
	return(scriptFile.path);
}
function getScriptName() {
	var fullpath = app.activeScript,
		scriptFile = new File(fullpath);
	return(scriptFile.name);
}


var messageWindow = null;
function messageClose() {
	if (!is_CS3_or_later) return;
	if (messageWindow != null) {
		try { messageWindow.close(); } catch(e){}
		messageWindow = null;
	}
}
function message(mess) {
	if (!is_CS3_or_later) return;
	if ( silentMode < 0 ) return;
	if ((mess == null) || (mess == "")) {
		messageClose();
		return;
	}
	try {
		if (messageWindow == null) messageCreate(mess);
		else {
			try { 
				messageWindow.messtext.text = mess;
			} catch(ex){}
		}
	} catch(e){}
	return;
}
function messageCreate(mess) {
	try {
		messageWindow = new Window('window', 'Export Progress', undefined, {maximizeButton:false, minimizeButton:false});
	} catch(e){}
	try {
		messageWindow.messageGroup = messageWindow.add('group');
		messageWindow.messageGroup.orientation = 'row';
		messageWindow.messageGroup.margins = [50,30,50,30];
		messageWindow.messtext = messageWindow.messageGroup.add('statictext', undefined, mess);
	} catch(e){}
	try {
		messageWindow.show();
	} catch(e){}
	return;
}


function listProperties(the_prop_to_query, show) {
	var props_arr = the_prop_to_query;
	var props = the_prop_to_query.reflect.properties;
	var pstr = "";
	for (var i = 0; i < props.length; i++) {
		pstr += "Property Name: '" + props[i].name + "'\n"
			  + "         Type: '" + typeof(props_arr[props[i].name]) + "' "
			  + "      Content: '" + props_arr[props[i].name] + "'\n";
		//pstr += "\n";
		if (props[i].name == "properties") listProperties(props[i], true);
	}
	if ((show == undefined) || (show == true)) alert(pstr);
	return(pstr);
}

function retrieve_issue_year_date(docname,fullpath) {
	// retrieve an issue date and year from document name if a number like 20080423 (YYYYMMDD) is contained.
	// otherwise, try to retrieve from export path
	//
	// assume we have exported the layouts from this path
	//		Macintosh HD:ePaperData:Xport:inSRC:COMPANY:OBJECT:YEAR:ISSUE:
	//   and the document has a naming convention like this SSS-YYYYMMDD-##-section
	//		where sss = a 2-3 digits shortcut for the object
	//		where YYYYMMDD = the issue date
	//		where ## = the page number


	// first, try get issue year and date from the document name
	// if we have no document date in the form YYYYMMDD - 8 following digits we have to try to get it from the path
	var s = "";
	// ok - lets try to find 8 following digits from the given 'docname'
	var found_a_date = false;
	var string_to_search = unescape(docname);
	var run = 1;
	var monthdayObj = new Object();
	monthdayObj.day = "";
	monthdayObj.month = "";

	if (fileDateFormat != "") {
		var adate = stringToDate(unescape(docname), fileDateFormat);
		if (adate != ",") {
			found_a_date = true
			var parts = adate.split(",");
			issueYear = parts[0];
			issueDate = parts[1];
		}
	}
	//alert("retrieve_issue_year_date()\nfileDateFormat: " + fileDateFormat + "\n\nissueYear: " + issueYear + "\nissueDate: " + issueDate);
	while ((found_a_date != true) && (run <= 3)) {
		//alert("run: " + run + "\n" + string_to_search);
		switch(run) {
			case 1:	// try to get full 8 digits date from document name
			case 2:	// try to get full 8 digits date from path name
				for (var i=0; i < string_to_search.length-7; i++) {
					if (	(string_to_search.charAt(i)!=" ") && (isNaN(string_to_search.charAt(i))==false)
							&& (isNaN(string_to_search.charAt(i+1))==false)
							&& (isNaN(string_to_search.charAt(i+2))==false)
							&& (isNaN(string_to_search.charAt(i+3))==false)
							&& (isNaN(string_to_search.charAt(i+4))==false)
							&& (isNaN(string_to_search.charAt(i+5))==false)
							&& (isNaN(string_to_search.charAt(i+6))==false)
							&& (isNaN(string_to_search.charAt(i+7))==false)
						) {
							found_a_date = true;
							issueYear = string_to_search.substr(i,4);
							issueDate = string_to_search.substr(i,8);
							break;
					}
				}
				break;
			case 3: 	// try to get 4 digits year from document name
				for (var i=0; i < string_to_search.length-3; i++) {
					if (	(string_to_search.charAt(i)!=" ") 
							&& (isNaN(string_to_search.charAt(i))==false)
							&& (isNaN(string_to_search.charAt(i+1))==false)
							&& (isNaN(string_to_search.charAt(i+2))==false)
							&& (isNaN(string_to_search.charAt(i+3))==false) ) {
						issueYear = string_to_search.substr(i,4);
						if ( (issueYear.indexOf(".") >= 0) || (parseInt(issueYear) > 2050) ) {
							issueYear = "";
							continue;
						}
						found_a_date = true;
						issueDate = "";
						// try to find a month in a docname like: le-provencal-de-paris-1908-11-15
						var nameparts = null;
						do {	// try to get a 2 digits month
							if (string_to_search.indexOf("_") > 0) {
								nameparts = string_to_search.split("_");
								get_issueMonthDay(nameparts,monthdayObj);
								if (monthdayObj.month != "") break;
							}
							if (string_to_search.indexOf("-") > 0) {
								nameparts = string_to_search.split("-");
								get_issueMonthDay(nameparts,monthdayObj);
								if (monthdayObj.month != "") break;
							}
							if (string_to_search.indexOf(" ") > 0) {
								nameparts = string_to_search.split(" ");
								get_issueMonthDay(nameparts,monthdayObj);
								if (monthdayObj.month != "") break;
							}
							if (string_to_search.indexOf("/") > 0) {
								nameparts = string_to_search.split("/");
								get_issueMonthDay(nameparts,monthdayObj);
								if (monthdayObj.month != "") break;
							}
						} while(false);
						break;
					}
				}
				break;
		}
		if (found_a_date == true) break;
		if (run > 3) break;
		switch (run) {
			case 1: string_to_search = unescape(fullpath);
				break;
			case 2: string_to_search = unescape(docname);
				break;
		}
		run++;
	}
	//alert("run: " + run + "\nissueDate: " + issueDate + "\nissueYear: " + issueYear + "\nfound from string: " + string_to_search);

	// get current date
	if ((found_a_date == false) || ((issueYear == "") || (issueDate == ""))) {
		var now = new Date();
		var year = "";
		if (found_a_date == false) year = "" + now.getFullYear();
		else year = issueYear;
		var month = "";
		if (monthdayObj.month != "") month = monthdayObj.month;
		else month = "" + (now.getMonth() + 1);
		if (month.length < 2) month = "0" + month;
		var day = "";
		if (monthdayObj.day != "") day = monthdayObj.day;
		else day = "" + now.getDate();
		if (day.length < 2) day = "0" + day;
		//alert("monthdayObj.month: " + monthdayObj.month + "\nmonthdayObj.day: " + monthdayObj.day
		//	+ "\nmonth: " + month + "\nday: " + day);
		issueYear = year;
		issueDate = year + month + day;
	}
	return null;
}

function get_issueMonthDay(nameparts,monthday) {	// get a 2 digits month from a string array
	if (nameparts == null) return("");
	var m = 0;
	for (m = 0; m < nameparts.length; m++) {	// try to get a 2 digits month
		if (nameparts[m].length != 2) continue;
		if ((nameparts[m].charAt(0)!=" ") && (isNaN(nameparts[m].charAt(0))==false)
				&& (nameparts[m].charAt(1)!=" ") && (isNaN(nameparts[m].charAt(1))==false)) {
				monthday.month = "" + nameparts[m];
				break;
		}
	}
	if ((m+1) >= (nameparts.length)) return;
	// get day in next part
	if ((nameparts[m + 1].charAt(0)!=" ") && (isNaN(nameparts[m + 1].charAt(0))==false)
			&& (nameparts[m + 1].charAt(1)!=" ") && (isNaN(nameparts[m + 1].charAt(1))==false)) {
			monthday.day = nameparts[m + 1].substr(0,2);
	}
	return;
}

function stringToDate(string, dateformat) {
	if (string == "") return(",");
	var matches = null;
	var mydateformat = dateformat.toUpperCase();
	if (mydateformat == "") mydateformat = "YYYYMMDD";
	var re = /(\d{4,4})(\d{2,2})(\d{2,2})/;
	switch (mydateformat) {
		case "YYYYMMDD":
		case "YYYYDDMM":
			re = /(\d{4,4})(\d{2,2})(\d{2,2})/;
			break;
		case "YYMMDD":
		case "YYDDMM":
		case "DDMMYY":
			re = /(\d{2,2})(\d{2,2})(\d{2,2})/;
			break;
		case "DDMMYYYY":
			re = /(\d{2,2})(\d{2,2})(\d{4,4})/;
			break;
		case "YYYY-MM-DD":
		case "YYYY-DD-MM":
			re = /(\d{4,4})-(\d{2,2})-(\d{2,2})/;
			break;
	}
	var myDate = null;
	var myDateStr = "";
	matches = string.match(re);
	if (matches == null) return(",");	// no match found
	switch (mydateformat) {
		default:
		case "YYYYMMDD":
			myDate = new Date(parseInt(matches[1],10), parseInt(matches[2] - 1,10), parseInt(matches[3],10));
			break;
		case "YYYYDDMM":
			myDate = new Date(parseInt(matches[1],10), parseInt(matches[3] - 1,10), parseInt(matches[2],10));
			break;
		case "YYMMDD":
			myDate = new Date(parseInt("20"+matches[1],10), parseInt(matches[2] - 1,10), parseInt(matches[3],10));
			break;
		case "YYDDMM":
			myDate = new Date(parseInt("20"+matches[1],10), parseInt(matches[3] - 1,10), parseInt(matches[2],10));
			break;
		case "DDMMYY":
			myDate = new Date(parseInt("20"+matches[3],10), parseInt(matches[2] - 1,10), parseInt(matches[1],10));
			break;
		case "DDMMYYYY":
			myDate = new Date(parseInt(matches[3],10), parseInt(matches[2] - 1,10), parseInt(matches[1],10));
			break;
		case "YYYY-MM-DD":
			myDate = new Date(parseInt(matches[1],10), parseInt(matches[2] - 1,10), parseInt(matches[3],10));
			break;
		case "YYYY-DD-MM":
			myDate = new Date(parseInt(matches[1],10), parseInt(matches[3] - 1,10), parseInt(matches[2],10));
			break;
	}
	//alert("mydateformat: " + mydateformat + "\n" + string + "\nfound matches:\n1: " + matches[1] + " -> " + parseInt(matches[1],10) + "\n2: " + matches[2] + " -> " + parseInt(matches[2],10) + "\n3: " + matches[3] + " -> " + parseInt(matches[3],10) + "\n\ndate: " + myDate.toString());
	if (myDate == null) return(",");	// no valid date

	var day = "" + myDate.getDate();
	if (day.length < 2) day = "0" + day;
	var month = "" + (myDate.getMonth() + 1);
	if (month.length < 2) month = "0" + month;
	var year = "" + myDate.getFullYear();
	while (year.length < 4) year = "0" + year;

	myDateStr = year + "," + year+ month + day;
	//alert("matches: " + matches + "\nmyDate: " + myDate.toString() + "\nmyDateStr: " + myDateStr);
	return myDateStr;
}

function retrieve_objectShortcut(docname) {	// get an id from document name
	if ( (docname == null) || (docname == "") ) return("");
	var mydocname = docname;
	if (mydocname.indexOf(".") >= 0) mydocname = mydocname.substr(0,mydocname.lastIndexOf("."));
	//alert("mydocname: '" + mydocname + "'");
	var start_underscore = mydocname.indexOf("_");
	var start_blank = mydocname.indexOf(" ");
	var start_divis = mydocname.indexOf("-");
	var start_dot = mydocname.indexOf(".");
	var end = mydocname.length;
	var end_was_set = false;
	if ((start_underscore < end) && (start_underscore >= 2)) { end = start_underscore; end_was_set = true; }
	if ((start_blank < end) && (start_blank >= 2)) { end = start_blank; end_was_set = true; }
	if ((start_divis < end) && (start_divis >= 2)) { end = start_divis; end_was_set = true; }
	if ((start_dot < end) && (start_dot >= 2)) { end = start_dot; end_was_set = true; }
	if (end_was_set == false) {
		if ((end < 2) || (end > 6)) end = 4;
	}
	if (end > mydocname.length) object_id = mydocname;
	else object_id = mydocname.substring(0,end);
	/*alert("start_underscore: " + start_underscore
		+ "\nstart_blank: " + start_blank
		+ "\nstart_divis: " + start_divis
		+ "\nstart_dot: " + start_dot
		+ "\n\nend: " + end
		+ "\nobject_id: " + object_id
		);*/
	return (object_id);
}


function startsWith (str,subs) { return str.indexOf(subs) === 0; }

function endsWith (str,subs) { 
	if (str.length < subs.length) return(false);
	if ((str == "") || (subs == "")) return(false);
	var endstr = str.substr(str.length - subs.length);
	return (endstr == subs);
}

function dotText(text, length) {
	if (text == null || text == "") return(text);
	if (length <= 5) return("");
	if (text.length <= length) return(text);
	var str1 = text.substring(0,parseInt(length/2)-3);
	var str2 = text.substring(text.length-length/2);
	return(str1 + "..." + str2);
}

// the functions xPlatformName MUST be equal to the ones in INXepaper.js (DO NOT CHANGE WITHOUT ADVICE OF THE PRODUCER!!!)
function xPlatformName(uri) {
	return(xPlatformName(uri,false));
}
function xPlatformName(uri,encode) {
	if ((typeof(uri) == 'undefined') || (uri == null) || (uri == "")) return("");
	var myuri = uri;
	// make sure to translate many invalid characters in file names like: ? [ ] / \ = + * < > : ; " ,
	try {
		myuri = myuri.replace(/\?/g,"x3F");
		myuri = myuri.replace(/\[/g,"x5B");
		myuri = myuri.replace(/\]/g,"x5D");
	  myuri = myuri.replace(/\//g,"x2F");
		myuri = myuri.replace(/\\/g,"x5C");
		myuri = myuri.replace(/\=/g,"x3D");
		myuri = myuri.replace(/\+/g,"x28");
		myuri = myuri.replace(/\*/g,"x2A");
		myuri = myuri.replace(/\</g,"x3C");
		myuri = myuri.replace(/\>/g,"x3E");
		myuri = myuri.replace(/\:/g,"x3A");
		myuri = myuri.replace(/\;/g,"x3B");
		myuri = myuri.replace(/\"/g,"x22");
		myuri = myuri.replace(/\,/g,"x2C");
		myuri = myuri.replace(/\'/g,"x27");
	
		myuri = myuri.replace(/#/g,"x23");
		myuri = myuri.replace(/\&/g,"x26");
		myuri = myuri.replace(/%/g,"x");

		if (encode) {
			myuri = encodeURI(myuri);
			myuri = myuri.replace(/%/g,'x');
		}
	} catch(e){}

	return (myuri);
}


/****************************
 * return file system name of full path/name
 */
function getFsName(path) {
	if ((path == null) || (path == "")) return("");
	if (path.toLowerCase().indexOf("/volumes/") == 0) return(path);	// OSX /Volumes
	if (path.toLowerCase().indexOf("//") == 0) return(path);	// UNC path
	if (path.toLowerCase().indexOf("smb://") == 0) return(path);	// UNC path
	var f = new File(path);
	return(f.fsName);
}

function getURIName(path) {
	if ((path == null) || (path == "")) return("");
	try {
		var f = new File(path);
		return(f.fullName);
	} catch(e) {}
	return(path);
}


/**************************
 * expand string with variables
 * variable names: $$varname$
 */
function add_vars(varstr) {
	if ((varstr == null) || (varstr == "")) return("");
	var vs = varstr;
	vs = vs.replace(/\$\$mainExportFolderOut\$/g,expandPath(mainExportFolderOut));
	vs = vs.replace(/\$\$BXSLT4InDesignBasePath\$/g,expandPath(BXSLT4InDesignBasePath));

	return (vs);
}
function expandPath(path) {
	if ((path == null) || (path == "")) return("");
	var fp = (new File (path)).fsName;
	fp = fp.replace(/ /g,"~blnk~");
	if ( endsWith(fp,folderSeparator) == false) fp += folderSeparator;
	//alert("fp: " + fp);
	return(fp);
}


/**************************
 * get language code
 */
function getAppLanguageCode(locale) {
	if ((locale == null) || (locale == "")) return("en");
	switch (locale) {	// all InDesign locales
		case Locale.ARABIC_LOCALE: return("ar");					// Arabic						1279476082 = 'LCAr'
		case Locale.CZECH_LOCALE: return("cz");						// Czech						1279476602 = 'LCCz'
		case Locale.DANISH_LOCALE: return("dn");					// Danish						1279476846 = 'LCDn'
		case Locale.ENGLISH_LOCALE: return("en");					// English						1279477102 = 'LCEn'
		case Locale.FINNISH_LOCALE: return("fn");					// Finnish						1279477358 = 'LCFn'
		case Locale.FRENCH_LOCALE: return("fr");					// French						1279477362 = 'LCFr'
		case Locale.GERMAN_LOCALE: return("de");					// German						1279477613 = 'LCGm'
		case Locale.GREEK_LOCALE: return("gr");						// Greek						1279477618 = 'LCGr'
		case Locale.HEBREW_LOCALE: return("hb");					// Hebrew						1279477858 = 'LCHb'
		case Locale.HUNGARIAN_LOCALE: return("hu");					// Hungarian					1279477877 = 'LCHu'
		case Locale.INTERNATIONAL_ENGLISH_LOCALE: return("en");		// International English		1279477097 = 'LCEi'
		case Locale.ITALIAN_LOCALE: return("it");					// Italian						1279478132 = 'LCIt'
		case Locale.JAPANESE_LOCALE: return("jp");					// Japanese						1279478384 = 'LCJp'
		case Locale.KOREAN_LOCALE: return("ko");					// en_KoreanLocale				1279478639 = 'LCKo'
		case Locale.POLISH_LOCALE: return("pl");					// Polish						1279479916 = 'LCPl'
		case Locale.PORTUGUESE_LOCALE: return("pg");				// Portuguese					1279479911 = 'LCPg'
		case Locale.ROMANIAN_LOCALE: return("ro");					// Romanian						1279480431 = 'LCRo'
		case Locale.RUSSIAN_LOCALE: return("ru");					// Russian						1279480437 = 'LCRu'
		case Locale.SIMPLIFIED_CHINESE_LOCALE: return("cn");		// simplified chinese			1279476590 = 'LCCn'
		case Locale.SPANISH_LOCALE: return("sp");					// Spanish						1279480688 = 'LCSp'
		case Locale.SWEDISH_LOCALE: return("sw");					// Swedish						1279480695 = 'LCSw'
		case Locale.TRADITIONAL_CHINESE_LOCALE: return("tw");		// traditional chinese			1279480951 = 'LCTw'
		case Locale.TURKISH_LOCALE: return("tr");					// Turkish						1279480946 = 'LCTr'
		case Locale.UKRAINIAN_LOCALE: return("uk");					// Ukrainian					1279481195 = 'LCUk' 
	}
	// default
	return("en");
}

function loc (which,par1,par2,par3,par4) {
	if (which >= lg.length) return("");
	try {
		if (lg[which].indexOf("%%") < 0) return(lg[which]);
	}
	catch(e) {
		alert("lg error for which: " + which);
		return("");
	}
	var tx = lg[which];
	if (par1) tx = tx.replace(/%%1%%/gi,par1);
	if (par2) tx = tx.replace(/%%2%%/gi,par2);
	if (par3) tx = tx.replace(/%%3%%/gi,par3);
	if (par4) tx = tx.replace(/%%4%%/gi,par4);
	return(tx);
}

